/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file BclLoader.hh
 *
 * \brief Declaration of BCL loader.
 *
 * \author Marek Balint
 * \author Mauricio Varea
 */


#ifndef BCL2FASTQ_CONVERSION_BCLLOADER_HH
#define BCL2FASTQ_CONVERSION_BCLLOADER_HH


#include <vector>

#include <boost/filesystem/path.hpp>
#include <boost/ptr_container/ptr_vector.hpp>

#include "layout/Layout.hh"
#include "data/BclFile.hh"
#include "data/CycleBCIFile.hh"
#include "data/PositionsFile.hh"
#include "data/FilterFile.hh"
#include "data/ControlFile.hh"
#include "conversion/BclBuffer.hh"
#include "conversion/Stage.hh"
#include "conversion/Task.hh"

#include <boost/shared_ptr.hpp>

namespace bcl2fastq {


namespace conversion {


/// \brief Task: Load BCL data into buffer.
class BclLoadTask : public Task
{
public:

    /// \brief Constructor.
    /// \param bclFile BCL file to be loaded.
    /// \param cycleBciFile Cycle BCI file.
    /// \param inputDir Path to input directory.
    /// \param aggregateTilesFlag All tiles aggregated into single file flag.
    /// \param laneInfo Lane meata data.
    /// \param cycleInfo Cycle meta data.
    /// \param tileInfo Tile meta data.
    /// \param clustersCount Number of clusters to be loaded.
    /// \param ignoreMissingBcls Ignore missing BCLs flag.
    /// \param bclIdx bcl index
    /// \param outputBuffer Buffer for data to be loaded to.
    BclLoadTask(
        data::BclFile &bclFile,
        data::CycleBCIFile &cycleBciFile,
        const boost::filesystem::path &inputDir,
        bool aggregateTilesFlag,
        const layout::LaneInfo &laneInfo,
        const layout::CycleInfo &cycleInfo,
        layout::LaneInfo::TileInfosContainer::const_iterator &tileInfo,
        const std::vector<common::ClustersCount>& clustersCounts,
        bool ignoreMissingBcls,
        size_t bclIdx,
        BclBufferVec& outputBuffer
    );

public:

    virtual bool execute(common::ThreadVector::size_type threadNum);

private:

    /// \brief BCL file to be loaded.
    data::BclFile &bclFile_;

    /// \brief Cycle BCI file.
    data::CycleBCIFile &cycleBciFile_;

    /// \brief Input directory path.
    const boost::filesystem::path &inputDir_;

    /// \brief ALl tiles aggregated into single file flag.
    bool aggregateTilesFlag_;

    /// \brief Lane meta data.
    const layout::LaneInfo &laneInfo_;

    /// \brief Cycle meta data.
    const layout::CycleInfo &cycleInfo_;

    /// \brief Tile meta data.
    layout::LaneInfo::TileInfosContainer::const_iterator tileInfoIter_;

    /// \brief Number of clusters to load.
    const std::vector<common::ClustersCount>& clustersCounts_;

    /// \brief Ignore missing BCLs flag.
    bool ignoreMissingBcls_;

    /// \brief Index into bcl buffer
    size_t bclIdx_;

    /// \brief Buffer to load data to.
    BclBufferVec &outputBuffer_;
};


/// \brief Task: Load positions data into buffer.
class PositionsLoadTask : public Task
{
public:

    /// \brief Constructor.
    /// \param positionsFile Positions file.
    /// \param intensitiesDir Path to intensities directory.
    /// \param aggregateTilesFlag All tiles aggregated into single file flag.
    /// \param laneInfo Lane meata data.
    /// \param tileInfo Tile meta data.
    /// \param clustersCount Number of clusters to be loaded.
    /// \param ignoreMissingPositions Ignore missing positions flag.
    /// \param outputBuffer Buffer for data to be loaded to.
    PositionsLoadTask(
        boost::shared_ptr<data::PositionsFile> &positionsFile,
        const boost::filesystem::path &intensitiesDir,
        bool aggregateTilesFlag,
        bool isPatternedFlowcell,
        const layout::LaneInfo &laneInfo,
        layout::LaneInfo::TileInfosContainer::const_iterator &tileInfo,
        const std::vector<common::ClustersCount>& clustersCounts,
        bool ignoreMissingPositions,
        BclBufferVec& outputBuffer,
        BclBuffer::PositionsContainer& patternedFlowcellPositions
    );

public:

    virtual bool execute(common::ThreadVector::size_type threadNum);

    bool execute(BclBuffer::PositionsContainer& outputBuffer,
                 common::ClustersCount clustersCount);

private:

    /// \brief Positions file.
    boost::shared_ptr<data::PositionsFile>& positionsFile_;

    /// \brief Intensities directory path.
    const boost::filesystem::path &intensitiesDir_;

    /// \brief All tiles aggregated into single file flag.
    bool aggregateTilesFlag_;

    /// \brief Single location file flag.
    bool isPatternedFlowcell_;

    /// \brief Lane meta data.
    const layout::LaneInfo &laneInfo_;

    /// \brief Tile meta data.
    layout::LaneInfo::TileInfosContainer::const_iterator tileInfoIter_;

    /// \brief Number of clusters to load.
    const std::vector<common::ClustersCount>& clustersCounts_;

    /// \brief Ignore missing positions flag.
    bool ignoreMissingPositions_;

    /// \brief Buffer to load data to.
    BclBufferVec &outputBuffer_;

    /// \brief Buffer for patterned flowcell positions
    BclBuffer::PositionsContainer& patternedFlowcellPositions_;
};


/// \brief Task: Load filter data into buffer.
class FilterLoadTask : public Task
{
public:

    /// \brief Constructor.
    /// \param filterFile Filter file.
    /// \param inputDir Path to input directory.
    /// \param aggregateTilesFlag All tiles aggregated into single file flag.
    /// \param laneInfo Lane meata data.
    /// \param tileInfo Tile meta data.
    /// \param clustersCount Number of clusters to be loaded.
    /// \param ignoreMissingFilters Ignore missing filters flag.
    /// \param outputBuffer Buffer for data to be loaded to.
    /// \param controlsOutputBuffer Buffer for controls data to be loaded to.
    FilterLoadTask(
        boost::shared_ptr<data::FilterFile> &filterFile,
        const boost::filesystem::path &inputDir,
        bool aggregateTilesFlag,
        const layout::LaneInfo &laneInfo,
        layout::LaneInfo::TileInfosContainer::const_iterator& tileInfo,
        const std::vector<common::ClustersCount>& clustersCounts,
        bool ignoreMissingFilters,
        BclBufferVec& outputBuffer
    );

public:

    virtual bool execute(common::ThreadVector::size_type threadNum);

private:

    /// \brief Read the aggregate tiles filter file.
    /// \param filePath Path to file.
    void readAggregateTilesFilterFile(const layout::TileInfo& tileInfo,
                                      common::ClustersCount clustersCount,
                                      BclBuffer& outputBuffer,
                                      const boost::filesystem::path& filePath = boost::filesystem::path());

    /// \brief Read the filter file for a single tile.
    /// \param filePath Path to file.
    void readTileFilterFile(const boost::filesystem::path& filePath,
                            const layout::TileInfo& tileInfo,
                            BclBuffer& outputBuffer);

    /// \brief Read the filter file.
    /// \param filterFile File to read from.
    void readFilterFile(data::FilterFile& filterFile,
                        const layout::TileInfo& tileInfo,
                        common::ClustersCount clustersCount,
                        BclBuffer& outputBuffer);

    /// \brief Validate the filter records
    /// \param recordsRead Number of records read from file.
    /// \param filePath Path to file.
    void validateFilterRecords(std::size_t                    recordsRead,
                               common::ClustersCount          clustersCount,
                               const boost::filesystem::path& filePath,
                               BclBuffer&                     outputBuffer);

    /// \brief Filter file.
    boost::shared_ptr<data::FilterFile>& filterFile_;

    /// \brief Input directory path.
    const boost::filesystem::path &inputDir_;

    /// \brief All tiles aggregated into single file flag.
    bool aggregateTilesFlag_;

    /// \brief Lane meta data.
    const layout::LaneInfo &laneInfo_;

    /// \brief Tile meta data.
    layout::LaneInfo::TileInfosContainer::const_iterator tileInfoIter_;

    /// \brief Number of clusters to load.
    const std::vector<common::ClustersCount>& clustersCounts_;

    /// \brief Ignore missing filters flag.
    bool ignoreMissingFilters_;

    /// \brief Buffer to load control data to.
    BclBufferVec& outputBuffer_;
};


class ControlLoadTask : public Task
{
public:
    ControlLoadTask(boost::shared_ptr<data::ControlFile> &controlFile,
        const boost::filesystem::path &inputDir,
        const layout::LaneInfo &laneInfo,
        layout::LaneInfo::TileInfosContainer::const_iterator &tileInfo,
        bool ignoreMissingControls,
        BclBufferVec& outputBuffer);

    virtual bool execute(common::ThreadVector::size_type threadNum);

private:
    /// \brief Filter file.
    boost::shared_ptr<data::ControlFile>& controlFile_;

    /// \brief Input directory path.
    const boost::filesystem::path &inputDir_;

    /// \brief Lane meta data.
    const layout::LaneInfo &laneInfo_;

    /// \brief Tile meta data.
    layout::LaneInfo::TileInfosContainer::const_iterator tileInfoIter_;

    /// \brief Ignore missing controls flag.
    bool ignoreMissingControls_;

    /// \brief Buffer to load data to.
    BclBufferVec& outputBuffer_;
};


/// \brief BCL loader.
class BclLoader : public SourceStage<BclBufferVec>
{
public:

    /// \brief Constructor.
    /// \param threadsCount Number of threads.
    /// \param outputMediator Output mediator.
    /// \param layout Flowcell layout.
    /// \param laneInfo Lane meata data.
    /// \param ignoreMissingBcls Ignore missing BCLs flag.
    /// \param ignoreMissingFilters Ignore missing filters flag.
    /// \param ignoreMissingPositions Ignore missing positions flag.
    /// \param inputDir Input directory.
    /// \param intensitiesDir Intensities directory.
    BclLoader(
        common::ThreadVector::size_type threadsCount,
        StageMediator<OutputBuffer> &outputMediator,
        const layout::Layout &layout,
        const layout::LaneInfo &laneInfo,
        bool ignoreMissingBcls,
        bool ignoreMissingFilters,
        bool ignoreMissingPositions,
        bool ignoreMissingControls,
        const boost::filesystem::path &inputDir,
        const boost::filesystem::path &intensitiesDir
    );

public:

    virtual bool preExecute();

    virtual bool postExecute();

private:

    void bclMismatchCount(std::string fileType,
                          common::CycleNumber cycleNumber,
                          BclBuffer::BclsContainer::value_type::size_type realSize,
                          BclBuffer::BclsContainer::value_type::size_type expectedSize);

    void createUniqueFakePositions(BclBuffer& outputBuffer,
                                   BclBuffer::BclsContainer::value_type::size_type bufferSize);

private:

    /// \brief BCL files container type definition.
    typedef boost::ptr_vector<data::BclFile> BclFilesContainer;

    /// \brief Cycle BCI files container type definition.
    typedef boost::ptr_vector<data::CycleBCIFile> CycleBciFilesContainer;

private:

    /// \brief Layout.
    const layout::Layout &layout_;

    /// \brief Current lane.
    const layout::LaneInfo &laneInfo_;

    /// \brief Ignore missing BCLs flag.
    bool ignoreMissingBcls_;

    /// \brief Ignore missing filters flag.
    bool ignoreMissingFilters_;

    /// \brief Ignore missing positions flag.
    bool ignoreMissingPositions_;

    /// \brief Ignore missing controls.
    bool ignoreMissingControls_;

    /// \brief Tile being currently processed.
    layout::LaneInfo::TileInfosContainer::const_iterator currentTileInfo_;

    /// \brief Input directory.
    boost::filesystem::path inputDir_;

    /// \brief Intensities directory.
    boost::filesystem::path intensitiesDir_;

    /// \brief BCL files (one per cycle).
    BclFilesContainer bclFiles_;

    /// \brief Cycle BCI files (one per cycle).
    CycleBciFilesContainer cycleBciFiles_;

    /// \brief Positions file.
    boost::shared_ptr<data::PositionsFile> positionsFile_;

    /// \brief Filter file.
    boost::shared_ptr<data::FilterFile> filterFile_;

    /// \brief Control file.
    boost::shared_ptr<data::ControlFile> controlFile_;

    /// \brief Positions for a patterned flowcell.
    BclBuffer::PositionsContainer patternedFlowcellPositions_;

    /// \brief Unique index per read if reading the Posisions file failed.
    data::PositionsFile::Record::ClusterCoordinate uniqueFailedReadIndex_;

    /// \brief Cluster counts for the current set of tiles.
    std::vector<common::ClustersCount> clustersCounts_;
};


} // namespace conversion


} // namespace bcl2fastq


#endif // BCL2FASTQ_CONVERSION_BCLLOADER_HH


