/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file Demultiplexer.hh
 *
 * \brief Declaration of demultiplexer.
 *
 * \author Marek Balint
 * \author Mauricio Varea
 */


#ifndef BCL2FASTQ_CONVERSION_DEMULTIPLEXER_HH
#define BCL2FASTQ_CONVERSION_DEMULTIPLEXER_HH


#include "layout/Layout.hh"
#include "layout/LaneInfo.hh"
#include "layout/BarcodeTranslationTable.hh"
#include "stats/DemultiplexingStatsXml.hh"
#include "stats/BarcodeStats.hh"
#include "conversion/BclBuffer.hh"
#include "conversion/Stage.hh"
#include "conversion/Task.hh"



namespace bcl2fastq
{
namespace conversion
{


/// \brief Task: Demultiplex.
class DemultiplexTask : public Task
{
public:

    /// \brief Demultiplexing statistics (one per thread). One stats::BarcodeStats per tile.
    typedef std::vector<stats::BarcodeStats> DemuxStats;

public:

    /// \brief Constructor.
    /// \param inputBuffer Input buffer.
    /// \param readsInfos Read infos.
    /// \param offsetBegin Starting offset.
    /// \param offsetEnd Ending offset (one past the end).
    /// \param barcodeTranslationTable Barcode translation table.
    /// \param tileInfosBegin tile info
    /// \numIndexReads number of index reads
    /// \includeNonPfClusters If true, include non PF clusters.
    /// \param demuxStats Demultiplexing statistics.
    /// \param outputBuffer Output buffer.
    DemultiplexTask(
        const BclBuffer &inputBuffer,
        const layout::ReadInfosContainer& readInfos,
        BclBuffer::BclsContainer::size_type offsetBegin,
        BclBuffer::BclsContainer::size_type offsetEnd,
        const layout::BarcodeTranslationTable &barcodeTranslationTable,
        layout::LaneInfo::TileInfosContainer::const_iterator tileInfosBegin,
        size_t numIndexReads,
        bool includeNonPfClusters,
        std::vector<DemultiplexTask::DemuxStats> &demuxStats,
        BclBuffer::SamplesContainer &outputBuffer
    );

public:

    virtual bool execute(common::ThreadVector::size_type threadNum);

private:

    /// \brief Input buffer.
    const BclBuffer &inputBuffer_;

    /// \brief Read infos.
    const layout::ReadInfosContainer& readInfos_;

    /// \brief Starting offset.
    const BclBuffer::BclsContainer::size_type offsetBegin_;

    /// \brief Ending offset (one past the end).
    const BclBuffer::BclsContainer::size_type offsetEnd_;

    /// \brief Barcode to sample translation table.
    const layout::BarcodeTranslationTable &barcodeTranslationTable_;

    /// \brief Reference to the first tile, used to calculate index.
    layout::LaneInfo::TileInfosContainer::const_iterator tileInfosBegin_;

    /// \brief Number of index reads.
    size_t numIndexReads_;

    /// \brief Include non-PF clusters
    bool includeNonPfClusters_;

    /// \brief Demultiplexing statistics.
    std::vector<DemultiplexTask::DemuxStats> &demuxStats_;

    /// \brief Output buffer.
    BclBuffer::SamplesContainer &outputBuffer_;
};


/// \brief Demultiplexer
class Demultiplexer : public IntermediateStage<BclBufferVec, BclBufferVec>
{
public:

    /// \brief Maximum number of allowed mismatches in a barcode component type definition.
    typedef std::size_t MismatchesCount;

    /// \brief Maximum allowed mismatches mismatches in a barcode component container type definition.
    typedef std::vector<MismatchesCount> BarcodeMismatchCountsContainer;

public:

    /// \brief Constructor.
    /// \param threadsCount Number of threads.
    /// \param inputMediator Input mediator.
    /// \param outputMediator Output mediator.
    /// \param layout Flowcell layout.
    /// \param laneInfo Lane meata data.
    /// \param barcodeTranslationTable Table to translate barcodes to samples.
    /// \includeNonPfClusters If true, include non PF clusters.
    /// \param interopDir Interop directory.
    Demultiplexer(
        common::ThreadVector::size_type threadsCount,
        StageMediator<InputBuffer> &inputMediator,
        StageMediator<OutputBuffer> &outputMediator,
        const layout::Layout &layout,
        const layout::LaneInfo &laneInfo,
        const layout::BarcodeTranslationTable &barcodeTranslationTable,
        bool includeNonPfClusters,
        DemultiplexTask::DemuxStats &summaryDemuxStats
    );

public:

    virtual bool preExecute();

    virtual bool postExecute();

private:

    void createTileStats(const layout::BarcodeTranslationTable::SampleMetadata &sampleMetadata);

private:

    /// \brief Layout.
    const layout::Layout &layout_;

    /// \brief Current lane.
    const layout::LaneInfo &laneInfo_;

    /// \brief Barcode to sample translation table.
    const layout::BarcodeTranslationTable &barcodeTranslationTable_;

    /// \brief Include non-PF clusters
    bool includeNonPfClusters_;

    /// \brief Number of threads.
    common::ThreadVector::size_type threadsCount_;

    /// \brief Demultiplexing statistics (summary).
    DemultiplexTask::DemuxStats &summaryDemuxStats_;

    /// \brief Demultiplexing statistics (one per thread).
    std::vector<DemultiplexTask::DemuxStats> demuxStats_;

    /// \brief Flag determining whether demultiplexing should take place.
    bool doDemultiplex_;

    /// \brief Number of index reads.
    size_t numIndexReads_;
};


} // namespace conversion
} // namespace bcl2fastq


#endif // BCL2FASTQ_CONVERSION_DEMULTIPLEXER_HH


