/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file Stage.hh
 *
 * \brief Declaration of a processing stage.
 *
 * \author Marek Balint
 */


#ifndef BCL2FASTQ_CONVERSION_STAGE_HH
#define BCL2FASTQ_CONVERSION_STAGE_HH


#include <boost/noncopyable.hpp>
#include <boost/thread/mutex.hpp>

#include "common/Threads.hh"
#include "conversion/Task.hh"
#include "conversion/TaskQueue.hh"
#include "conversion/StageMediator.hh"


namespace bcl2fastq {


namespace conversion {


/// \brief Processing stage.
class Stage : private boost::noncopyable
{
public:

    /// \brief Constructor.
    /// \param threadsCount Number of threads.
    /// \param stageName Name of the stage.
    explicit Stage(common::ThreadVector::size_type threadsCount,
                   const std::string&              stageName);

    /// \brief Virtual destructor.
    virtual ~Stage() = 0;

public:

    /// \brief Run the processing stage.
    void run();

protected:

    /// \brief Get task queue.
    /// \return Task queue.
    TaskQueue & getTaskQueue();

private:

    /// \brief Thread function for stage execution.
    /// \param threadNum Thread number.
    void executeThreadFunction(common::ThreadVector::size_type threadNum);

    /// \brief Get next task.
    /// \return Pointer to the next task to be executed. In case there is no
    /// task left in the queue, @c NULL pointer will be returned.
    Task * getNextTask();

private:

    /// \brief Obtain data to work on.
    /// \retval true Data successfully obtained.
    /// \retval false No more data, work is done.
    /// \note Default action is NOOP. This function should be overridden
    /// by consumer processing stages.
    virtual bool getWork();

    /// \brief Submit result data.
    /// \note Default action is NOOP. This function should be overridden
    /// by producer processing stages.
    virtual void submitWork();

    /// \brief Announce that work has been done.
    /// \note Default action is NOOP. This function should be overridden
    /// by producer processing stages, in order to let consumer processing
    /// stage know that there is no more data coming its way.
    virtual void finishWork();

    /// \brief Prepare execution.
    /// \retval true Execution has been prepared and can start.
    /// \retval false Execution should not be started (e.g. work is done).
    /// \note There is no default action. This function should be overridden
    /// by processing stages to initialize working data and task queue.
    virtual bool preExecute() = 0;

    /// \brief Finish execution.
    /// \retval true Next iteration can start.
    /// \retval false Do not continue with next iteration (e.g. work is done).
    /// \note There is no default action. This function should be overridden
    /// by processing stages to do whatever clean-up and bookkeeping is needed
    /// after execution has been finished.
    virtual bool postExecute() = 0;

    /// \brief Announce that an error has occurred and thread is going down.
    /// \note There is no default action. This function should be overridden
    /// by processing stages to let other threads that this thread is going
    /// down and no more data will be neither consumed nor produced by it.
    virtual void terminate() = 0;

private:

    /// \brief total time spent in pre-execution step
    size_t preExecuteTime_;

    /// \brief total time spent in post-execution step
    size_t postExecuteTime_;

    /// \brief Synchronization mutex.
    boost::mutex mutex_;

    /// \brief Thread vector.
    common::ThreadVector threadVector_;

    /// \brief Task queue.
    TaskQueue taskQueue_;

    /// \brief Name of stage
    std::string stageName_;
};


/// \brief Source processing stage.
template<typename OutputBufferType>
class SourceStage : public Stage
{
public:

    /// \brief Output buffer type definition.
    typedef OutputBufferType OutputBuffer;

public:

    /// \brief Constructor.
    /// \param threadsCount Number of threads.
    /// \param stageName Name of the stage.
    /// \param outputMediator Output mediator.
    SourceStage(
        common::ThreadVector::size_type threadsCount,
        const std::string&              stageName,
        StageMediator<OutputBuffer>&    outputMediator
    );

    virtual ~SourceStage() { }

protected:

    /// \brief Get output buffer.
    /// \return Output buffer.
    typename SourceStage<OutputBufferType>::OutputBuffer & getOutputBuffer();

private:

    virtual void submitWork();

    virtual void finishWork();

    virtual void terminate();

private:

    /// \brief Output buffer.
    OutputBuffer outputBuffer_;

    /// \brief Output mediator.
    StageMediator<OutputBuffer> &outputMediator_;
};


/// \brief Sink processing stage.
template<typename InputBufferType>
class SinkStage : public Stage
{
public:

    /// \brief Input buffer type definition.
    typedef InputBufferType InputBuffer;

public:

    /// \brief Constructor.
    /// \param threadsCount Number of threads.
    /// \param inputMediator Input mediator.
    SinkStage(
        common::ThreadVector::size_type threadsCount,
        const std::string&              stageName,
        StageMediator<InputBuffer>&     inputMediator
    );

    virtual ~SinkStage() { }

protected:

    /// \brief Get input buffer.
    /// \return Input buffer.
    typename SinkStage<InputBufferType>::InputBuffer & getInputBuffer();

private:

    virtual bool getWork();

    virtual void terminate();

private:

    /// \brief Input buffer.
    InputBuffer inputBuffer_;

    /// \brief Input mediator.
    StageMediator<InputBuffer> &inputMediator_;
};


/// \brief Intermediate processing stage.
/// \todo Refactoring: Consider inheriting @c IntermediateStage from @c SourceStage
/// and @c SinkStage instead of duplicating all the code.
template<typename InputBufferType, typename OutputBufferType>
class IntermediateStage : public Stage
{
public:

    /// \brief Input buffer type definition.
    typedef InputBufferType InputBuffer;

    /// \brief Output buffer type definition.
    typedef OutputBufferType OutputBuffer;

public:

    /// \brief Constructor.
    /// \param threadsCount Number of threads.
    /// \param stageName Name of the stage
    /// \param inputMediator Input mediator.
    /// \param outputMediator Output mediator.
    IntermediateStage(
        common::ThreadVector::size_type threadsCount,
        const std::string&              stageName,
        StageMediator<InputBuffer>&     inputMediator,
        StageMediator<OutputBuffer>&    outputMediator
    );

    virtual ~IntermediateStage() { }

protected:

    /// \brief Get input buffer.
    /// \return Input buffer.
    typename IntermediateStage<InputBufferType, OutputBufferType>::InputBuffer & getInputBuffer();
    const typename IntermediateStage<InputBufferType, OutputBufferType>::InputBuffer & getInputBuffer() const;

    /// \brief Get output buffer.
    /// \return Output buffer.
    typename IntermediateStage<InputBufferType, OutputBufferType>::OutputBuffer & getOutputBuffer();

private:

    virtual bool getWork();

    virtual void submitWork();

    virtual void finishWork();

    virtual void terminate();

private:

    /// \brief Input buffer.
    InputBuffer inputBuffer_;

    /// \brief Input mediator.
    StageMediator<InputBuffer> &inputMediator_;

    /// \brief Output buffer.
    OutputBuffer outputBuffer_;

    /// \brief Output mediator.
    StageMediator<OutputBuffer> &outputMediator_;
};


} // namespace conversion


} // namespace bcl2fastq


#include "conversion/Stage.hpp"


#endif // BCL2FASTQ_CONVERSION_STAGE_HH


