/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file Stage.hpp
 *
 * \brief Implementation of a processing stage.
 *
 * \author Marek Balint
 */


#ifndef BCL2FASTQ_CONVERSION_STAGE_HPP
#define BCL2FASTQ_CONVERSION_STAGE_HPP


#include "conversion/Stage.hh"


namespace bcl2fastq {


namespace conversion {


template<typename OutputBufferType>
SourceStage<OutputBufferType>::SourceStage(
    common::ThreadVector::size_type threadsCount,
    const std::string&              stageName,
    StageMediator<OutputBuffer>&    outputMediator
)
: Stage(threadsCount,
        stageName)
, outputMediator_(outputMediator)
{
}

template<typename OutputBufferType>
typename SourceStage<OutputBufferType>::OutputBuffer & SourceStage<OutputBufferType>::getOutputBuffer()
{
    return outputBuffer_;
}

template<typename OutputBufferType>
void SourceStage<OutputBufferType>::submitWork()
{
    outputMediator_.submitWork(outputBuffer_);
}

template<typename OutputBufferType>
void SourceStage<OutputBufferType>::finishWork()
{
    outputMediator_.producerDone();
}

template<typename OutputBufferType>
void SourceStage<OutputBufferType>::terminate()
{
    outputMediator_.terminate();
}


template<typename InputBufferType>
SinkStage<InputBufferType>::SinkStage(
    common::ThreadVector::size_type threadsCount,
    const std::string&              stageName,
    StageMediator<InputBuffer>&     inputMediator
)
: Stage(threadsCount,
        stageName)
, inputMediator_(inputMediator)
{
}

template<typename InputBufferType>
typename SinkStage<InputBufferType>::InputBuffer & SinkStage<InputBufferType>::getInputBuffer()
{
    return inputBuffer_;
}

template<typename InputBufferType>
bool SinkStage<InputBufferType>::getWork()
{
    return inputMediator_.getWork(inputBuffer_);
}

template<typename InputBufferType>
void SinkStage<InputBufferType>::terminate()
{
    return inputMediator_.terminate();
}


template<typename InputBufferType, typename OutputBufferType>
IntermediateStage<InputBufferType, OutputBufferType>::IntermediateStage(
    common::ThreadVector::size_type threadsCount,
    const std::string&              stageName,
    StageMediator<InputBuffer>&     inputMediator,
    StageMediator<OutputBuffer>&    outputMediator
)
: Stage(threadsCount,
        stageName)
, inputMediator_(inputMediator)
, outputMediator_(outputMediator)
{
}

template<typename InputBufferType, typename OutputBufferType>
typename IntermediateStage<InputBufferType, OutputBufferType>::InputBuffer & IntermediateStage<InputBufferType, OutputBufferType>::getInputBuffer()
{
    return inputBuffer_;
}

template<typename InputBufferType, typename OutputBufferType>
const typename IntermediateStage<InputBufferType, OutputBufferType>::InputBuffer & IntermediateStage<InputBufferType, OutputBufferType>::getInputBuffer() const
{
    return inputBuffer_;
}

template<typename InputBufferType, typename OutputBufferType>
typename IntermediateStage<InputBufferType, OutputBufferType>::OutputBuffer & IntermediateStage<InputBufferType, OutputBufferType>::getOutputBuffer()
{
    return outputBuffer_;
}

template<typename InputBufferType, typename OutputBufferType>
bool IntermediateStage<InputBufferType, OutputBufferType>::getWork()
{
    return inputMediator_.getWork(inputBuffer_);
}

template<typename InputBufferType, typename OutputBufferType>
void IntermediateStage<InputBufferType, OutputBufferType>::submitWork()
{
    outputMediator_.submitWork(outputBuffer_);
}

template<typename InputBufferType, typename OutputBufferType>
void IntermediateStage<InputBufferType, OutputBufferType>::finishWork()
{
    outputMediator_.producerDone();
}

template<typename InputBufferType, typename OutputBufferType>
void IntermediateStage<InputBufferType, OutputBufferType>::terminate()
{
    inputMediator_.terminate();
    outputMediator_.terminate();
}


} // namespace task


} // namespace bcl2fastq


#endif // BCL2FASTQ_CONVERSION_STAGE_HPP


