/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file StageMediator.hh
 *
 * \brief Declaration of a mediator for processing stages.
 *
 * \author Marek Balint
 */


#ifndef BCL2FASTQ_CONVERSION_STAGEMEDIATOR_HH
#define BCL2FASTQ_CONVERSION_STAGEMEDIATOR_HH


#include <boost/noncopyable.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/condition_variable.hpp>


namespace bcl2fastq {


namespace conversion {


/// \brief Processing stages mediator.
template<typename BufferType>
class StageMediator : private boost::noncopyable
{
public:

    /// \brief Buffer type.
    typedef BufferType Buffer;

public:

    /// \brief Default constructor.
    StageMediator();

public:

    /// \brief Get data to consume.
    /// \param buffer Buffer to be filled in with the data.
    /// \retval true There is unfinished work to be done.
    /// \retval false There is no outstanding work to be done.
    bool getWork(Buffer &buffer);

    /// \brief Submit produced data.
    /// \param buffer Buffer containing produced results.
    void submitWork(Buffer &buffer);

    /// \brief Announce that producer finished all its work.
    void producerDone();

    /// \brief Announce that an error has occured and thread is going down.
    void terminate();

private:

    /// \brief Producer pre-swap synchronization flag.
    bool producerPreSwap_;

    /// \brief Producer post-swap synchronization flag.
    bool producerPostSwap_;

    /// \brief Consumer pre-swap synchronization flag.
    bool consumerPreSwap_;

    /// \brief Consumer post-swap synchronization flag.
    bool consumerPostSwap_;

    /// \brief Producer finished synchronization flag.
    bool producerFinished_;

    /// \brief Terminate flag.
    bool terminate_;

    /// \brief Buffer to be swapped.
    Buffer *buffer_;

    /// \brief Synchronization mutex.
    boost::mutex mutex_;

    /// \brief Synchronization condition variable.
    boost::condition_variable conditionVariable_;
};


} // namespace conversion


} // namespace bcl2fastq


#include "conversion/StageMediator.hpp"


#endif // BCL2FASTQ_CONVERSION_STAGEMEDIATOR_HH


