/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file StageMediator.hpp
 *
 * \brief Implementation of a mediator for processing stages.
 *
 * \author Marek Balint
 */


#ifndef BCL2FASTQ_CONVERSION_STAGEMEDIATOR_HPP
#define BCL2FASTQ_CONVERSION_STAGEMEDIATOR_HPP


#include <boost/thread/locks.hpp>

#include "common/Logger.hh"
#include "common/Debug.hh"
#include "conversion/StageMediator.hh"


namespace bcl2fastq {


namespace conversion {


template<typename BufferType>
StageMediator<BufferType>::StageMediator()
: producerPreSwap_(false)
, producerPostSwap_(false)
, consumerPreSwap_(false)
, consumerPostSwap_(false)
, producerFinished_(false)
, terminate_(false)
, buffer_(NULL)
, mutex_()
, conditionVariable_()
{
}

template<typename BufferType>
bool StageMediator<BufferType>::getWork(Buffer &buffer)
{
    boost::unique_lock<boost::mutex> lock(mutex_);
    if (terminate_)
    {
        return false;
    }

    buffer_ = &buffer;

    producerPreSwap_ = true;
    conditionVariable_.notify_all();
    while (!consumerPreSwap_ && !producerFinished_)
    {
        if (terminate_)
        {
            return false;
        }
        conditionVariable_.wait(lock);
    }
    if (producerFinished_)
    {
        BCL2FASTQ_LOG(common::LogLevel::DEBUG) << "Consumer done" << std::endl;
        return false;
    }
    consumerPreSwap_ = false;

    producerPostSwap_ = true;
    conditionVariable_.notify_all();
    while (!consumerPostSwap_)
    {
        if (terminate_)
        {
            return false;
        }
        conditionVariable_.wait(lock);
    }
    consumerPostSwap_ = false;

    return true;
}

template<typename BufferType>
void StageMediator<BufferType>::submitWork(Buffer &buffer)
{
    boost::unique_lock<boost::mutex> lock(mutex_);
    if (terminate_)
    {
        return;
    }

    consumerPreSwap_ = true;
    conditionVariable_.notify_all();
    while (!producerPreSwap_)
    {
        if (terminate_)
        {
            return;
        }
        conditionVariable_.wait(lock);
    }
    producerPreSwap_ = false;

    BCL2FASTQ_ASSERT_MSG(buffer_ != NULL, "Pointer to consumer's buffer not initialized");
    using std::swap;
    swap(*buffer_, buffer);

    consumerPostSwap_ = true;
    conditionVariable_.notify_all();
    while (!producerPostSwap_)
    {
        if (terminate_)
        {
            return;
        }
        conditionVariable_.wait(lock);
    }
    producerPostSwap_ = false;
}


template<typename BufferType>
void StageMediator<BufferType>::producerDone()
{
    boost::unique_lock<boost::mutex> lock(mutex_);
    BCL2FASTQ_LOG(common::LogLevel::DEBUG) << "Producer done" << std::endl;
    producerFinished_ = true;
    conditionVariable_.notify_all();
}

template<typename BufferType>
void StageMediator<BufferType>::terminate()
{
    boost::unique_lock<boost::mutex> lock(mutex_);
    BCL2FASTQ_LOG(common::LogLevel::DEBUG) << "Stage terminated" << std::endl;
    terminate_ = true;
    conditionVariable_.notify_all();
}


} // namespace conversion


} // namespace bcl2fastq


#endif // BCL2FASTQ_CONVERSION_STAGEMEDIATOR_HPP


