/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file Demultiplexer.cpp
 *
 * \brief Implementation of demultiplexer.
 *
 * \author Marek Balint
 * \author Mauricio Varea
 */


#include <algorithm>
#include <numeric>
#include <utility>

#include <boost/foreach.hpp>
#include <boost/assign/list_of.hpp>


#include "common/FileSystem.hh"
#include "layout/Barcode.hh"
#include "conversion/BclBaseConversion.hh"
#include "conversion/Converter.hh"
#include "conversion/Demultiplexer.hh"
#include "conversion/FastqIterator.hh"


namespace bcl2fastq
{
namespace conversion
{


DemultiplexTask::DemultiplexTask(
    const BclBuffer &inputBuffer,
    const layout::ReadInfosContainer& readInfos,
    BclBuffer::BclsContainer::size_type offsetBegin,
    BclBuffer::BclsContainer::size_type offsetEnd,
    const layout::BarcodeTranslationTable &barcodeTranslationTable,
    layout::LaneInfo::TileInfosContainer::const_iterator tileInfosBegin,
    size_t numIndexReads,
    bool includeNonPfClusters,
    std::vector<DemultiplexTask::DemuxStats> &demuxStats,
    BclBuffer::SamplesContainer &outputBuffer
)
: Task()
, inputBuffer_(inputBuffer)
, readInfos_(readInfos)
, offsetBegin_(offsetBegin)
, offsetEnd_(offsetEnd)
, barcodeTranslationTable_(barcodeTranslationTable)
, tileInfosBegin_(tileInfosBegin)
, numIndexReads_(numIndexReads)
, includeNonPfClusters_(includeNonPfClusters)
, demuxStats_(demuxStats)
, outputBuffer_(outputBuffer)
{
}

bool DemultiplexTask::execute(common::ThreadVector::size_type threadNum)
{
    for (BclBuffer::BclsContainer::size_type offset = offsetBegin_; offset < offsetEnd_; ++offset)
    {
        const bool filterFlag = inputBuffer_.filters_.at(offset).data_;

        layout::Barcode& barcode = outputBuffer_[offset].second;

        // Clear the barcode. This does not free memory, which is good.
        // We don't want to reallocate the string every time, which 
        // is particularly bad since this is multithreaded and the memory allocation will block.
        barcode.reset(numIndexReads_);

        size_t i = 0;
        layout::ReadInfo::CycleInfosContainer::const_iterator::difference_type previousReadsLength = 0;
        for (const auto& readInfo : readInfos_)
        {
            size_t currentReadLength = readInfo.cycleInfos().size();
            if (readInfo.isIndexRead() && currentReadLength != 0)
            {
                const FastqConstIterator begin(inputBuffer_.bcls_.begin() + previousReadsLength, offset);
                const FastqConstIterator end(inputBuffer_.bcls_.begin() + previousReadsLength + currentReadLength, offset);

                size_t dist = end-begin;
                std::string& barcodeString = barcode.components_[i].getString();
                barcodeString.resize(dist);
                for (size_t j = 0; j < dist; ++j)
                {
                    barcodeString[j] = convertBcl2FastqBase(begin[j]);
                }
                ++i;
            }
            previousReadsLength += readInfo.cyclesToLoad().size();
        }

        unsigned int mismatches(0);
        layout::BarcodeTranslationTable::SampleMetadata sampleMetadata = barcodeTranslationTable_.translateBarcode(barcode,mismatches);

        outputBuffer_.at(offset).first = sampleMetadata;

        if (filterFlag || includeNonPfClusters_)
        {
            stats::BarcodeStats barcodeStats( sampleMetadata, inputBuffer_.tileInfo_ - tileInfosBegin_ );
            const std::pair<DemuxStats::iterator, DemuxStats::iterator> statsRecord = std::equal_range(
                demuxStats_.at(threadNum).begin(),
                demuxStats_.at(threadNum).end(),
                barcodeStats
            );
            BCL2FASTQ_ASSERT_MSG(statsRecord.first != statsRecord.second, "Demultiplexing statistics record for sample index " << sampleMetadata.sampleIndex_ << " and barcode index " << sampleMetadata.barcodeIndex_ << " not found");
            statsRecord.first->incrementBarcodeCount(mismatches);
        }
    }

    return true;
}


Demultiplexer::Demultiplexer(
    common::ThreadVector::size_type threadsCount,
    StageMediator<InputBuffer> &inputMediator,
    StageMediator<OutputBuffer> &outputMediator,
    const layout::Layout &layout,
    const layout::LaneInfo &laneInfo,
    const layout::BarcodeTranslationTable &barcodeTranslationTable,
    bool includeNonPfClusters,
    DemultiplexTask::DemuxStats &summaryDemuxStats
)
: IntermediateStage<BclBufferVec, BclBufferVec>(threadsCount, "Demultiplexing", inputMediator, outputMediator)
, layout_(layout)
, laneInfo_(laneInfo)
, barcodeTranslationTable_(barcodeTranslationTable)
, includeNonPfClusters_(includeNonPfClusters)
, threadsCount_(threadsCount)
, summaryDemuxStats_(summaryDemuxStats)
, demuxStats_()
, doDemultiplex_(false)
, numIndexReads_(0)
{
    layout::LaneInfo::SampleInfosContainer::size_type sampleIndex = 0;
    BOOST_FOREACH (const layout::SampleInfo &sampleInfo, std::make_pair(laneInfo_.sampleInfosBegin(), laneInfo_.sampleInfosEnd()))
    {
        layout::BarcodesContainer::size_type barcodesCount = sampleInfo.getBarcodes().size();
        if( 0==barcodesCount )
        {
            createTileStats( layout::BarcodeTranslationTable::SampleMetadata(sampleIndex) );
        } else {
            for( layout::BarcodesContainer::size_type barcodeIndex = 0;
                 barcodeIndex < barcodesCount;
                 ++barcodeIndex )
            {
                createTileStats( layout::BarcodeTranslationTable::SampleMetadata(sampleIndex,barcodeIndex) );
            }
        }
        ++sampleIndex;
    }

    for (common::ThreadVector::size_type threadNum = 0; threadNum < threadsCount_; ++threadNum)
    {
        demuxStats_.push_back(summaryDemuxStats_);
    }

    if (laneInfo.sampleInfosEnd() - laneInfo.sampleInfosBegin() > 1)
    {
        for (const auto& readInfo : laneInfo.readInfos())
        {
            if ((readInfo.isIndexRead()) && !readInfo.cycleInfos().empty())
            {
                ++numIndexReads_;
                doDemultiplex_ = true;
            }
        }
    }
    else
    {
        BCL2FASTQ_ASSERT_MSG( laneInfo.sampleInfosEnd() - laneInfo.sampleInfosBegin() == 1,
                              "There must be at least default sample" );
        // in case there is sample with barcode defined in sample sheet, there will be also default sample (i.e. at least 2 samples in total) and this branch will never get executed
        BCL2FASTQ_ASSERT_MSG( laneInfo.sampleInfosBegin()->getBarcodes().empty(),
                              "There should be either default sample with no barcode or the only sample defined in sample sheet without barcode");
    }
}

void Demultiplexer::createTileStats(const layout::BarcodeTranslationTable::SampleMetadata &sampleMetadata)
{
    layout::LaneInfo::TileInfosContainer::size_type tilesCount = laneInfo_.getTileInfos().size();
    for( layout::LaneInfo::TileInfosContainer::size_type tileIndex = 0;
         tileIndex < tilesCount;
         ++tileIndex )
    {
        summaryDemuxStats_.push_back(DemultiplexTask::DemuxStats::value_type(sampleMetadata,tileIndex));
    }
}

bool Demultiplexer::preExecute()
{
    OutputBuffer &inputBuffer = this->getInputBuffer();
    OutputBuffer &outputBufferVec = this->getOutputBuffer();
    TaskQueue &taskQueue = this->getTaskQueue();

    using std::swap;
    swap(inputBuffer, outputBufferVec);

    for (auto& outputBuffer : outputBufferVec)
    {
        outputBuffer.samples_.clear();
        BCL2FASTQ_ASSERT_MSG(outputBuffer.bcls_.size() > 0, "No BCL cycles to process");
        outputBuffer.samples_.resize(outputBuffer.bcls_.front().size());

        if (doDemultiplex_)
        {
            const BclBuffer::BclsContainer::size_type dataSize = outputBuffer.bcls_.front().size();
            for (BclBuffer::BclsContainer::size_type offset = 0; offset < dataSize; offset += ClustersPerTask)
            {
                taskQueue.addTask(new DemultiplexTask(
                    outputBuffer,
                    laneInfo_.readInfos(),
                    offset,
                    std::min(offset+ClustersPerTask, dataSize),
                    barcodeTranslationTable_,
                    laneInfo_.getTileInfos().begin(),
                    numIndexReads_,
                    includeNonPfClusters_,
                    demuxStats_,
                    outputBuffer.samples_
                ));
            }
        }
    }

    return true;
}

bool Demultiplexer::postExecute()
{
    for (common::ThreadVector::size_type threadNum = 0; threadNum < threadsCount_; ++threadNum)
    {
        DemultiplexTask::DemuxStats::size_type demuxIndex = 0;
        BOOST_FOREACH (stats::BarcodeStats &barcodeStats, std::make_pair(demuxStats_.at(threadNum).begin(), demuxStats_.at(threadNum).end()))
        {
            stats::BarcodeStats &summaryBarcodeStats = summaryDemuxStats_.at(demuxIndex);
            BCL2FASTQ_ASSERT_MSG(!(barcodeStats < summaryBarcodeStats), "Inconsistent demultiplexing statistics structures");
            BCL2FASTQ_ASSERT_MSG(!(summaryBarcodeStats < barcodeStats), "Inconsistent demultiplexing statistics structures");
            summaryBarcodeStats += barcodeStats;
            barcodeStats.reset();
            ++demuxIndex;
        }
    }

    return true;
}


} // namespace conversion
} // namespace bcl2fastq


