/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file FakeFastqProducer.cpp
 *
 * \brief Implementation of fake FASTQ producer.
 *
 * \author Marek Balint
 */


#include "conversion/FakeFastqProducer.hh"
#include "conversion/SampleIndex.hh"


namespace bcl2fastq {


namespace conversion {


/// \brief Length of one FASTQ sequence.
static const SampleIndex::FastqOffsetsContainer::size_type fastqLength = 32;

/// \brief Number of FASTQ chunks per sample and read to be generatoed.
static const FastqBuffer::FastqsContainer::value_type::value_type::size_type fastqTasksCount = 8;


FakeFastqProducer::FakeFastqProducer(
    common::ThreadVector::size_type threadsCount,
    StageMediator<OutputBuffer> &outputMediator,
    const layout::Layout &layout,
    const layout::LaneInfo &laneInfo,
    bool createFastqsForIndexReads
)
: SourceStage<FastqBuffer>(threadsCount, "Fake fastq producing", outputMediator)
, layout_(layout)
, laneInfo_(laneInfo)
, createFastqsForIndexReads_(createFastqsForIndexReads)
{
}


bool FakeFastqProducer::preExecute()
{
    FastqBuffer &outputBuffer = this->getOutputBuffer();

    {
        const layout::LaneInfo::SampleInfosContainer::const_iterator sampleInfosBegin = laneInfo_.sampleInfosBegin();
        const layout::LaneInfo::SampleInfosContainer::const_iterator sampleInfosEnd = laneInfo_.sampleInfosEnd();
        layout::LaneInfo::SampleInfosContainer::size_type sampleInfoIndex = 0;
        outputBuffer.fastqs_.resize(sampleInfosEnd - sampleInfosBegin);
        BOOST_FOREACH (const layout::SampleInfo &sampleInfo, std::make_pair(sampleInfosBegin, sampleInfosEnd))
        {
            (void) sampleInfo;
            outputBuffer.fastqs_.at(sampleInfoIndex).resize(laneInfo_.readInfos().size());
            layout::ReadInfosContainer::size_type readInfoIndex = 0;
            for (const auto& readInfo : laneInfo_.readInfos())
            {
                if (readInfo.isDataRead() || (createFastqsForIndexReads_ && readInfo.isIndexRead()))
                {
                    outputBuffer.fastqs_.at(sampleInfoIndex).at(readInfoIndex).resize(fastqTasksCount);
                    for (FastqBuffer::FastqsContainer::value_type::value_type::size_type taskIndex = 0; taskIndex < fastqTasksCount; ++taskIndex)
                    {
                        outputBuffer.fastqs_.at(sampleInfoIndex).at(readInfoIndex).at(taskIndex).resize(ClustersPerTask*fastqLength);
                    }
                }
                ++readInfoIndex;
            }
            ++sampleInfoIndex;
        }
    }

    return true;
}

bool FakeFastqProducer::postExecute()
{
    return true;
}


} // namespace conversion


} // namespace bcl2fastq


