/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file TaskQueue.cpp
 *
 * \brief Implementation of task queue.
 *
 * \author Marek Balint
 */


#include "common/Debug.hh"
#include "conversion/TaskQueue.hh"


namespace bcl2fastq {


namespace conversion {


TaskQueue::TaskQueue()
: boost::ptr_vector<Task>()
, currentTaskIndex_(0)
{
}

void TaskQueue::addTask(Task *task)
{
    // Documentation for boost::ptr_sequence_adapter (parent of boost::ptr_vector):
    // - void push_back( T* x );
    //     - Requirements: x != 0
    //     - Effects: Inserts the pointer into container and takes ownership of it
    //     - Throws: bad_pointer if x == 0
    //     - Exception safety: Strong guarantee
    BCL2FASTQ_ASSERT_MSG(task, "The pointer passed in does not point to the valid task");
    this->push_back(task);
}

Task * TaskQueue::getTask()
{
    if (!(currentTaskIndex_ < this->size()))
    {
        return NULL;
    }
    return &this->at(currentTaskIndex_);
}

void TaskQueue::advanceTask()
{
    BCL2FASTQ_ASSERT_MSG(currentTaskIndex_ < this->size(), "Already at the end of the task queue");
    ++currentTaskIndex_;
}

TaskQueue::size_type TaskQueue::getOutstandingTasksCount() const
{
    return this->size() - currentTaskIndex_;
}

void TaskQueue::clear()
{
    this->boost::ptr_vector<Task>::clear();
    boost::ptr_vector<Task>().swap(*this);
    currentTaskIndex_ = 0;
}


} // namespace task


} // namespace bcl2fastq


