import argparse
from typing import List, Optional

import runner
from runner import ProcessCmd


def execute_joint(
        ref_tar: str,
        common_dragen_args: List[str],
        gvcfs: List[str],
        tn_tsvs: List[str],
        bam_inputs: List[str],
        small_var_options: str,
        cnv_options: str,
        sv_options: str,
        sv_bed: Optional[str],
) -> None:
    """Executes DRAGEN joint genotyping / calling

    :param common_dragen_args: Required DRAGEN args include:
        --output-file-prefix
        --pedigree-file
    """
    process_cmds = runner.create_setup_ref_cmds(ref_tar)
    if gvcfs:
        process_cmds.append(ProcessCmd(runner.create_dragen_cmd([
            *common_dragen_args,
            small_var_options,
            *[f'--variant {gvcf}' for gvcf in gvcfs],
            '--enable-joint-genotyping true',
        ])))
    if tn_tsvs:
        process_cmds.append(ProcessCmd(runner.create_dragen_cmd([
            *common_dragen_args,
            cnv_options,
            *[f'--cnv-input {tn_tsv}' for tn_tsv in tn_tsvs],
            '--enable-cnv true',
        ])))
    if bam_inputs:
        process_cmds.append(ProcessCmd(runner.create_dragen_cmd([
            *common_dragen_args,
            sv_options,
            *[f'--bam-input {bam_input}' for bam_input in bam_inputs],
            '--enable-sv true',
            f'--sv-call-regions-bed {sv_bed}' if sv_bed else '',
        ])))

    runner.run_dragen_cmds(process_cmds)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--gvcfs', nargs='*', default=[], type=str)
    parser.add_argument('--tn-tsvs', nargs='*', default=[], type=str)
    parser.add_argument('--bam-inputs', nargs='*', default=[], type=str)
    parser.add_argument('--small-var-options', default='', type=str)
    parser.add_argument('--cnv-options', default='', type=str)
    parser.add_argument('--sv-options', default='', type=str)
    parser.add_argument('--sv-call-regions-bed', type=str)
    parser.add_argument('--ref-tar', type=str)
    parser.add_argument('--debug', action='store_true')
    args, extras = parser.parse_known_args()

    runner.execute_with_no_traceback(
        lambda: execute_joint(
            args.ref_tar,
            extras,
            args.gvcfs,
            args.tn_tsvs,
            args.bam_inputs,
            args.small_var_options,
            args.cnv_options,
            args.sv_options,
            args.sv_call_regions_bed
        ),
        args.debug
    )


if __name__ == '__main__':
    main()
