/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file BclBuffer.hh
 *
 * \brief Declaration of BCL buffer.
 *
 * \author Marek Balint
 */


#ifndef BCL2FASTQ_CONVERSION_BCLBUFFER_HH
#define BCL2FASTQ_CONVERSION_BCLBUFFER_HH


#include <vector>

#include "data/PositionsFile.hh"
#include "data/FilterFile.hh"
#include "data/ControlFile.hh"
#include "layout/Layout.hh"
#include "layout/LaneInfo.hh"
#include "layout/BarcodeTranslationTable.hh"

namespace bcl2fastq {


namespace conversion {

static const size_t ClustersPerTask = 16384;
 
/// \brief BCL buffer.
struct BclBuffer
{
public:

    /// \brief BCL cycles data container type definition.
    typedef std::vector<std::vector<char> > BclsContainer;

    /// \brief Cluster to sample assignment data container type definition.
    typedef std::vector< std::pair<layout::BarcodeTranslationTable::SampleMetadata, layout::Barcode> > SamplesContainer;

    /// \brief Position data container type definition.
    typedef std::vector<data::PositionsFile::Record> PositionsContainer;

    /// \brief Filter data container type definition.
    typedef std::vector<data::FilterFile::Record> FiltersContainer;

    /// \brief Control data container type definition.
    typedef std::vector<data::ControlFile::Record> ControlsContainer;

public:

    /// \brief Buffers for cycles data.
    BclsContainer bcls_;

    /// \brief Cluster to sample assignments.
    SamplesContainer samples_;

    /// \brief Tile meta data.
    layout::LaneInfo::TileInfosContainer::const_iterator tileInfo_;

    /// \brief Buffer for position data.
    PositionsContainer positions_;

    /// \brief Buffer for filter data.
    FiltersContainer filters_;

    /// \brief Buffer for control data.
    ControlsContainer controls_;

public:

    /// \brief Number of BCL clusters to be processed in single execution iteration.
    /// \todo Refactoring: Put this into CMake configuration.
    static const BclsContainer::value_type::size_type clustersPerChunk_;
};

typedef std::vector<BclBuffer> BclBufferVec;

/// \brief Swap function for BCL buffer.
/// \param lhs Left-hand-side parameter.
/// \param rhs Right-hand-side parameter.
void swap(BclBuffer &lhs, BclBuffer &rhs);


} // namespace conversion


} // namespace bcl2fastq


#endif // BCL2FASTQ_CONVERSION_BCLBUFFER_HH


