/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file FastqIterator.hh
 *
 * \brief Declaration of FASTQ iterator.
 *
 * \author Marek Balint
 */


#ifndef BCL2FASTQ_CONVERSION_FASTQITERATOR_HH
#define BCL2FASTQ_CONVERSION_FASTQITERATOR_HH


#include <vector>
#include <iterator>

#include <boost/iterator/iterator_facade.hpp>
#include <boost/iterator/reverse_iterator.hpp>
#include <boost/mpl/if.hpp>

#include "conversion/BclBuffer.hh"


namespace bcl2fastq {


namespace conversion {


/// \brief FASTQ iterator.
/// \note As Boost::Iterator documentation notes, the superior way to implement
/// this should be by using @c boost::iterator_adaptor. However, that class
/// is flawed, because it allows interoperability of completely unrelated
/// iterators (@c boost::is_convertible is @c true for @c T and @c const @c T,
/// but also for e.g. @c char and @c int).
template<bool IsConstant>
class FastqIteratorTemplate : public boost::iterator_facade<
    FastqIteratorTemplate<IsConstant>,   // Derived
    typename boost::mpl::if_c<           // Value
    IsConstant,
    const char,
    char
    >::type,
    boost::random_access_traversal_tag  // CategoryOrTraversal
>
{
public:

    /// \brief Default constructor.
    /// \post Iterator is singular.
    FastqIteratorTemplate()
        : iterator_()
        , offset_(0)
    {
    }

    /// \brief Constructor.
    /// \param iterator BCL chunk iterator.
    /// \param offset Offset from beginning of the BCL chunk.
    FastqIteratorTemplate(
        typename boost::mpl::if_c<
        IsConstant,
        BclBuffer::BclsContainer::const_iterator,
        BclBuffer::BclsContainer::iterator
        >::type iterator,
        BclBuffer::BclsContainer::value_type::size_type offset
        )
        : iterator_(iterator)
        , offset_(offset)
    {
    }

    /// \brief Conversion operator to constant iterator.
    operator FastqIteratorTemplate<true>() const
    {
        return FastqIteratorTemplate<true>(
            iterator_,
            offset_
            );
    }

private:

    friend class boost::iterator_core_access;

    /// \brief Dereference core operation.
    /// \return Reference to element being pointed to.
    /// \pre Iterator is dereferencable.
    typename FastqIteratorTemplate<IsConstant>::reference dereference() const
    {
        return iterator_->begin()[offset_];
    }

    /// \brief Equality comparison core operation.
    /// \param rhs Right-hand-side parameter.
    /// \retval true Left-hand-side and right-hand-side parameters are equal.
    /// \retval false Left-hand-side and right-hand-side parameters are not equal.
    template<bool IsRhsConstant>
    bool equal(const FastqIteratorTemplate<IsRhsConstant> &rhs) const
    {
        return (iterator_ == rhs.iterator_) && (offset_ == rhs.offset_);
    }

    /// \brief Increment core operation.
    /// \pre Iterator is dereferencable.
    /// \post Iterator points to the next element.
    void increment()
    {
        ++iterator_;
    }

    /// \brief Decrement core operation.
    /// \pre Iterator is dereferencable.
    /// \post Iterator points to the next element.
    void decrement()
    {
        --iterator_;
    }

    /// \brief Advance core operation.
    /// \param diff Distance by which the iterator is to be advanced.
    /// \pre Iterator is dereferencable.
    /// \post Iterator has been advanced by specified distance.
    void advance(typename FastqIteratorTemplate<IsConstant>::difference_type diff)
    {
        iterator_ += diff;
    }

    /// \brief Distance core operation.
    /// \param rhs Right-hand-side parameter.
    /// \return Distance between this iterator and right-hand-side iterator.
    template<bool IsRhsConstant>
    typename FastqIteratorTemplate<IsConstant>::difference_type distance_to(const FastqIteratorTemplate<IsRhsConstant> &rhs) const
    {
        return rhs.iterator_ - iterator_;
    }

private:

    /// \brief BCL chunk iterator.
    typename boost::mpl::if_c<IsConstant, BclBuffer::BclsContainer::const_iterator, BclBuffer::BclsContainer::iterator>::type iterator_;

    /// \brief Offset from beginning of the BCL chunk.
    BclBuffer::BclsContainer::value_type::size_type offset_;
};


/// \brief FASTQ iterator.
typedef FastqIteratorTemplate<false> FastqIterator;

/// \brief FASTQ reverse iterator.
typedef boost::reverse_iterator<FastqIterator> FastqReverseIterator;

/// \brief FASTQ constant iterator.
typedef FastqIteratorTemplate<true> FastqConstIterator;

/// \brief FASTQ constant reverse iterator.
typedef boost::reverse_iterator<FastqConstIterator> FastqConstReverseIterator;


} // namespace conversion


} // namespace bcl2fastq


#endif // BCL2FASTQ_CONVERSION_FASTQITERATOR_HH
