/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file FastqWriter.hh
 *
 * \brief Declaration of FASTQ writer.
 *
 * \author Marek Balint
 */


#ifndef BCL2FASTQ_CONVERSION_FASTQWRITER_HH
#define BCL2FASTQ_CONVERSION_FASTQWRITER_HH


#include "layout/Layout.hh"
#include "data/FastqFile.hh"
#include "conversion/FastqBuffer.hh"
#include "conversion/Stage.hh"
#include "conversion/Task.hh"


namespace bcl2fastq {


namespace conversion {


/// \brief Task: Write FASTQs from buffer to file.
class FastqWriteTask : public Task
{
public:

    /// \brief FASTQ files container type definition.
    typedef boost::ptr_vector<data::FastqFile> FastqFilesContainer;

    /// \brief FASTQ file external state with metadata type definition.
    struct FastqFileStateInfo
    {
    public:

        /// \brief Default constructor.
        FastqFileStateInfo();

        /// \brief Constructor.
        explicit FastqFileStateInfo(data::FastqFile::FastqFileStatePtr fastqFileStatePtr);

    public:

        /// \brief Allocation flag.
        bool allocationFlag_;

        /// \brief FASTQ file external state.
        data::FastqFile::FastqFileStatePtr fastqFileStatePtr_;
    };

public:

    /// \brief Constructor.
    /// \param fastqFiles FASTQ files (one per thread).
    /// \param fastqFileStateInfo External state and metadata of FASTQ file to be written to.
    /// \param inputBuffer Input FASTQ data buffer to be read from.
    FastqWriteTask(
        FastqFilesContainer &fastqFiles,
        FastqFileStateInfo &fastqFileStateInfo,
        const FastqBuffer::FastqsContainer::value_type::value_type &inputBuffer
    );

public:

    virtual bool execute(common::ThreadVector::size_type threadNum);

private:

    /// \brief FASTQ files (one per thread).
    FastqFilesContainer &fastqFiles_;

    /// \brief File state meta data of the FASTQ file to be written to.
    FastqFileStateInfo &fastqFileStateInfo_;

    /// \brief Buffer to read data from.
    const FastqBuffer::FastqsContainer::value_type::value_type &inputBuffer_;
};


/// \brief FASTQ writer.
class FastqWriter : public SinkStage<FastqBuffer>
{
public:

    /// \brief Constructor.
    /// \param threadsCount Number of threads.
    /// \param inputMediator Input mediator.
    /// \param layout Flowcell layout.
    /// \param laneInfo Lane meata data.
    /// \param createFastqsForIndexReads Create FASTQ files also for index reads flag.
    /// \param outputDir Output directory.
    FastqWriter(
        common::ThreadVector::size_type threadsCount,
        StageMediator<InputBuffer> &inputMediator,
        const layout::Layout &layout,
        const layout::LaneInfo &laneInfo,
        bool createFastqsForIndexReads,
        const boost::filesystem::path& outputDir,
        bool noLaneSplitting
    );

public:

    virtual bool preExecute();

    virtual bool postExecute();

private:

    /// \brief FASTQ file external states container type definition.
    typedef std::vector<std::vector<FastqWriteTask::FastqFileStateInfo> > FastqFileStatesContainer;

private:
// Class to delete an empty file on destruction
class EmptyFileDeleter
{
public:
    EmptyFileDeleter() : filePaths_() { }
    ~EmptyFileDeleter();

    void addFile(boost::filesystem::path& filePath) { filePaths_.push_back(filePath); }

private:
    std::vector<boost::filesystem::path> filePaths_;
};

    /// \brief Layout.
    const layout::Layout &layout_;

    /// \brief Delete empty files on destruction
    EmptyFileDeleter emptyFileDeleter_;

    /// \brief FASTQ files (one per thread).
    FastqWriteTask::FastqFilesContainer fastqFiles_;

    /// \brief FASTQ file states.
    FastqFileStatesContainer fastqFileStates_;
};


} // namespace conversion


} // namespace bcl2fastq


#endif // BCL2FASTQ_CONVERSION_FASTQWRITER_HH


