/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file Task.hh
 *
 * \brief Declaration of a workload item a.k.a. task.
 *
 * \author Marek Balint
 */


#ifndef BCL2FASTQ_CONVERSION_TASK_HH
#define BCL2FASTQ_CONVERSION_TASK_HH


#include "common/Threads.hh"


namespace bcl2fastq {


namespace conversion {


/// \brief Task.
class Task
{
public:

    /// \brief (Default) constructor.
    /// \param stickyFlag Sticky flag value.
    /// \note Sticky tasks are not removed from task queue upon execution,
    /// which means that multiple theads may (and under most circumstances will)
    /// be executing the same instance of task at the same time. Therefore
    /// @c execute function of sticky task must be thread-safe.
    explicit Task(bool stickyFlag = false);

    /// \brief Virtual destructor.
    virtual ~Task() = 0;

public:

    /// \brief Execute the task.
    /// \param threadNum Thread number.
    /// \retval true Continue with execution on current thread.
    /// \retval false Finish current execution iteration on current thread.
    /// \note This function should be overriden in order for task to actually
    /// do something.
    virtual bool execute(common::ThreadVector::size_type threadNum) = 0;

    /// \brief Get sticky flag value.
    /// \return Sticky flag value.
    bool isSticky() const;

protected:

    /// \brief Set sticky flag value.
    /// \param value New value of sticky flag to be set.
    /// \see Note on sticky tasks and thread-safety in the constructor.
    void setStickyFlag(bool value);

private:

    /// \brief Sticky flag.
    bool stickyFlag_;
};


/// \brief Separator task.
class SeparatorTask : public Task
{
public:

    /// \brief Default constructor.
    SeparatorTask();

public:

    virtual bool execute(common::ThreadVector::size_type);
};


} // namespace conversion


} // namespace bcl2fastq


#endif // BCL2FASTQ_CONVERSION_TASK_HH


