/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file BclFile.hh
 *
 * \brief Declaration of BCL file.
 *
 * \author Marek Balint
 * \author Mauricio Varea
 */


#ifndef BCL2FASTQ_DATA_BCLFILE_HH
#define BCL2FASTQ_DATA_BCLFILE_HH


#include <memory>

#include <boost/filesystem/path.hpp>

#include "io/FileBufWithReopen.hh"
#include "io/GzipDecompressor.hh"
#include "common/Types.hh"
#include "data/FileReader.hh"


namespace bcl2fastq {


namespace data {


/// \brief BCL file.
class BclFile : public BinaryFileReader
{
    typedef boost::error_info<struct tag_errmsg, std::string> errmsg_info;
public:

    /// \brief Default constructor.
    BclFile();

public:
    /// \brief Open BCL file (aggregate tiles).
    /// \param inputDir Directory to look for the bcl file.
    /// \param laneNumber The lane number.
    /// \param cycleNumber The cycle number.
    /// \param ignoreErrors Supress errors opening file and/or reading its header.
    void openFile(
        const boost::filesystem::path& inputDir,
        common::LaneNumber             laneNumber,
        common::CycleNumber            cycleNumber,
        bool                           ignoreErrors
    );

    /// \brief Open BCL file (single tile).
    /// \param inputDir Directory to look for the bcl file.
    /// \param laneNumber The lane number.
    /// \param tileNumber The tile number.
    /// \param cycleNumber The cycle number.
    /// \param ignoreErrors Supress errors opening file and/or reading its header.
    void openFile(
        const boost::filesystem::path& inputDir,
        common::LaneNumber             laneNumber,
        common::TileNumber             tileNumber,
        common::CycleNumber            cycleNumber,
        bool                           ignoreErrors
    );

    /// \brief Verify the bcl file exists and return the name. (single tile).
    /// \param inputDir Directory to look for the bcl file.
    /// \param laneNumber The lane number.
    /// \param tileNumber The tile number.
    /// \param cycleNumber The cycle number.
    /// \param ignoreErrors Supress errors opening file and/or reading its header.
    /// \param fileName Output parameter for the bcl file name.
    static void getAndVerifyFileName(const boost::filesystem::path& inputDir,
                                     common::LaneNumber             laneNumber,
                                     common::TileNumber             tileNumber,
                                     common::CycleNumber            cycleNumber,
                                     bool                           ignoreErrors,
                                     boost::filesystem::path&       fileName);

    /// \brief Verify the bcl file exists and return the name. (aggregate tiles).
    /// \param inputDir Directory to look for the bcl file.
    /// \param laneNumber The lane number.
    /// \param cycleNumber The cycle number.
    /// \param ignoreErrors Supress errors opening file and/or reading its header.
    /// \param fileName Output parameter for the bcl file name.
    static void getAndVerifyFileName(const boost::filesystem::path& inputDir,
                                     common::LaneNumber             laneNumber,
                                     common::CycleNumber            cycleNumber,
                                     bool                           ignoreErrors,
                                     boost::filesystem::path&       fileName);

    /// \brief Read bytes from file to buffer.
    /// \param targetBuffer Target buffer to read to.
    /// \param targetSize Maximum number of bytes to be read.
    /// \pre <tt>this->isOpen() == true</tt>
    /// \return Number of bytes read.
    std::streamsize read(char *targetBuffer, std::streamsize targetSize);

    /// \brief Seek to specified position.
    /// \param compressedOffset Offset in compressed data denoting start of BGZF block.
    /// \param uncompressedOffset Offset in uncompressed data of the BGZF block.
    void seek(std::streamsize compressedOffset, std::streamsize uncompressedOffset);

private:
    /// \brief Open BCL file (ignore errors).
    /// \param path Path to the BCL file to be opened.
    /// \param ignoreErrors Supress errors opening file and/or reading its header.
    /// \param compressed Indicates whether the file is compressed or not.
    /// \param defaultClustersCount Number of clusters to assume in case of error opening the file
    /// and/or reading its header.
    void openFile(
        const boost::filesystem::path &path,
        bool ignoreErrors,
        bool compressed,
        common::ClustersCount defaultClustersCount = 0
    );

    std::streamsize read(std::istream &is, char *targetBuffer, std::streamsize targetSize);

    /// \brief Gets the file type string used for error messages.
    /// \return File type string
    virtual std::string getFileTypeStr() const { return "BCL"; }

    /// \brief Reading position in the file.
    io::FileBufWithReopen::pos_type currentFilePosition_;

    /// \brief GZip decompression filter.
    io::GzipDecompressor gzipDecompressor_;

    /// \brief Whether we need to use the decompressor for reading or not.
    bool isCompressed_;
};


} // namespace data


} // namespace bcl2fastq


#endif // BCL2FASTQ_DATA_BCLFILE_HH


