/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file FastqCreator.cpp
 *
 * \brief Implementation of FASTQ creator.
 *
 * \author Marek Balint
 * \author Mauricio Varea
 */


#include <string>
#include <algorithm>
#include <iterator>
#include <utility>

#include <boost/format.hpp>
#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <boost/assign/list_of.hpp>

#include "common/FastIo.hh"
#include "layout/BarcodeTranslationTable.hh"
#include "io/GzipCompressor.hh"

#include "conversion/BclBaseConversion.hh"
#include "conversion/FastqCreator.hh"

namespace bcl2fastq
{
namespace conversion
{

namespace detail
{


} // namespace detail

FastqCreateTask::FastqCreateTask(
    const BclBufferVec &bclBuffers,
    size_t cycleIndex,
    size_t cycleIndexEnd,
    const layout::FlowcellInfo &flowcellInfo,
    const layout::LaneInfo &laneInfo,
    const layout::ReadInfo& currentReadInfo,
    SampleIndex::FastqOffsetsContainer::const_iterator offsetsBegin,
    SampleIndex::FastqOffsetsContainer::const_iterator offsetsEnd,
    std::size_t maskShortAdapterReads,
    float adapterStringency,
    const boost::ptr_vector<AdapterLocator>& maskAdapters,
    const boost::ptr_vector<AdapterLocator>& trimAdapters,
    bool generateReverseComplementFastqs,
    bool includeNonPfClusters,
    bool useBgzf,
    int compressionLevel,
    bool findAdaptersWithSlidingWindow,
    std::vector< FastqCreateTask::ConversionStats > &tileStats,
    std::vector<stats::BarcodeHits> &unknownBarcodes,
    FastqBuffer::FastqsContainer::value_type::value_type::value_type &outputBuffer
)
: Task()
, bclBuffers_(bclBuffers)
, cycleIndex_(cycleIndex + currentReadInfo.getBclBufferOffset())
, cycleIndexEnd_(cycleIndex_ + currentReadInfo.cycleInfos().size())
, flowcellInfo_(flowcellInfo)
, laneInfo_(laneInfo)
, readInfo_(currentReadInfo)
, sampleMetadata_(bclBuffers_[offsetsBegin->first].samples_[offsetsBegin->second].first)
, offsetsBegin_(offsetsBegin)
, offsetsEnd_(offsetsEnd)
, outputBuffer_(outputBuffer)
, buffer_()
, useBgzf_(useBgzf)
, compressionLevel_(compressionLevel)
, findAdaptersWithSlidingWindow_(findAdaptersWithSlidingWindow)
, minimumTrimmedReadLength_(laneInfo_.getMinimumTrimmedReadLength())
, maskShortAdapterReads_(maskShortAdapterReads)
, adapterStringency_(adapterStringency)
, maskAdapters_(maskAdapters)
, trimAdapters_(trimAdapters)
, generateReverseComplementFastqs_(generateReverseComplementFastqs)
, includeNonPfClusters_(includeNonPfClusters)
, tileStats_(tileStats)
, unknownBarcodes_(unknownBarcodes)
, umiCycles_()
{
    common::CycleNumber umiCycleOffset = 0;
    for (const auto& readInfo : laneInfo_.readInfos())
    {
        // We do not load BCL files for cycles we don't use.
        // The index into the BCL buffer needs to account for this.

        const common::CycleRange& umiCycles = readInfo.getUmiCycles();
        if (umiCycles.second != 0)
        {
            // These are the indexes into the bcl buffer
            umiCycles_.push_back(common::CycleRange(umiCycles.first + umiCycleOffset, umiCycles.second + umiCycleOffset));
        }

        umiCycleOffset += readInfo.cyclesToLoad().size();
    }
}

bool FastqCreateTask::execute(common::ThreadVector::size_type threadNum)
{
    io::GzipCompressor compressor(outputBuffer_,
                                  useBgzf_,
                                  boost::iostreams::gzip_params(compressionLevel_, // config param with default=4
                                                                boost::iostreams::zlib::deflated, // default
                                                                15, // default
                                                                9));

    std::for_each(
        offsetsBegin_,
        offsetsEnd_,
        boost::bind(
            &FastqCreateTask::fastqCreate,
            this,
            threadNum,
            _1,
            boost::ref(compressor)
        )
    );

    return true;
}


namespace detail {


static const char fastqHeaderStart = '@';
static const char fastqHeaderSeparator = '\n';
static const char fastqHeaderValuesSeparator = ':';
static const char fastqHeaderPositionsSeparator = ' ';
static const char fastqHeaderUmiSeparator = '+';
static const std::string fastqBasesSeparator("\n+\n");
static const char fastqQualitiesSeparator('\n');
static const char fastqMaskedBase('N');
static const char fastqMaskedQuality('#');


template< typename StatsType >
static typename std::vector<StatsType>::iterator findStats( std::vector<StatsType> &stats, StatsType init )
{
    typedef typename std::vector<StatsType>::iterator ReturnType;
    const std::pair< ReturnType, ReturnType > statsRecord = std::equal_range( stats.begin(), stats.end(), init );
    BCL2FASTQ_ASSERT_MSG(statsRecord.first != statsRecord.second,
                        "Conversion statistics record for sample #" << init.index.first.sampleIndex_
                        << ", barcode #" << init.index.first.barcodeIndex_
                        << ", tile #" << init.index.second << " not found");
    return statsRecord.first;
}

template< typename StatsType >
static void transferStats( std::vector< typename StatsType::value_type > &stats,
                           std::vector< typename StatsType::value_type > &summary )
{
    typename StatsType::size_type conversionIndex = 0;
    BOOST_FOREACH( typename StatsType::value_type &tileStats, std::make_pair(stats.begin(), stats.end()) )
    {
        typename StatsType::value_type &summaryTileStats = summary[conversionIndex];
        BCL2FASTQ_ASSERT_MSG(!(tileStats < summaryTileStats), "Inconsistent conversion statistics structures");
        BCL2FASTQ_ASSERT_MSG(!(summaryTileStats < tileStats), "Inconsistent conversion statistics structures");
        summaryTileStats += tileStats;
        tileStats.reset();
        ++conversionIndex;
    }
}


} // namespace detail


bool FastqCreateTask::computeStatistics( common::ThreadVector::size_type threadNum,
                                         FastqConstIterator basesBegin,
                                         FastqConstIterator basesEnd,
                                         SampleIndex::FastqOffsetsContainer::value_type offset,
                                         bool filterFlag,
                                         size_t trimmedCount )
{
    layout::LaneInfo::TileInfosContainer::size_type tileIndex =  std::distance( laneInfo_.getTileInfos().begin(),
                                                                                bclBuffers_[offset.first].tileInfo_ );
    ConversionStats::first_type::iterator allReads = detail::findStats(
        tileStats_[threadNum].first,
        stats::TileBarcodeStats( stats::TileStats<stats::AllReadsStats>::Index( sampleMetadata_, tileIndex ))
    );
    ConversionStats::second_type::iterator perRead = detail::findStats(
        tileStats_[threadNum].second,
        stats::ReadBarcodeStats( stats::TileStats<stats::ReadStats>::Index( sampleMetadata_, tileIndex )
                               , readInfo_.getNumber() )
    );
    stats::ReadStats readStats;
    readStats.yield = std::distance(basesBegin, basesEnd);
    readStats.yieldQ30 = std::count_if(basesBegin, basesEnd,boost::bind(&compareBcl2FastqQuality,_1,30));
    readStats.qualityScoreSum = std::accumulate(
                                     basesBegin,
                                     basesEnd,
                                     common::ClustersCount(0),
                                     boost::bind(
                                         std::plus<common::ClustersCount>(),
                                         _1,
                                         boost::bind(&convertBcl2FastqRawQuality, _2)
                                     )
                                );
    if( 1 == readInfo_.getNumber() )  { (*allReads)[stats::TileBarcodeStats::RAW].clusterCount += 1; }
    (*allReads)[stats::TileBarcodeStats::RAW].RecordAdapterStats(trimmedCount, readInfo_.getNumber());
    (*perRead)[stats::TileBarcodeStats::RAW] += readStats;
    if( filterFlag )
    {
        if( 1 == readInfo_.getNumber() )  { (*allReads)[stats::TileBarcodeStats::PF].clusterCount += 1; }
        (*allReads)[stats::TileBarcodeStats::PF].RecordAdapterStats(trimmedCount, readInfo_.getNumber());
        (*perRead)[stats::TileBarcodeStats::PF] += readStats;
    }
    return (0 != sampleMetadata_.sampleIndex_);
}


void FastqCreateTask::fastqCreate(
    common::ThreadVector::size_type threadNum,
    SampleIndex::FastqOffsetsContainer::value_type offset,
    io::GzipCompressor& compressor
)
{
    const FastqConstIterator begin(bclBuffers_[offset.first].bcls_.begin() + cycleIndex_, offset.second);
    const FastqConstIterator end(bclBuffers_[offset.first].bcls_.begin() + cycleIndexEnd_, offset.second);

    const bool filterFlag = bclBuffers_[offset.first].filters_[offset.second].data_;

    size_t trimmedCount = 0;
    if (includeNonPfClusters_ || filterFlag)
    {
        buffer_.clear();

        createHeader(offset,
                     filterFlag);

        createBasesAndQualities(offset,
                                compressor,
                                trimmedCount);
    }

    if (readInfo_.isDataRead())
    {
        if (!computeStatistics(threadNum, begin, end, offset, filterFlag, trimmedCount))
        {
            if( 1 == readInfo_.getNumber() && (includeNonPfClusters_ || filterFlag) )  { unknownBarcodes_[threadNum].record( bclBuffers_[offset.first].samples_[offset.second].second ); }
        }
    }
}

void FastqCreateTask::writeHeaderElement(const std::string& element)
{
    size_t bufferSize = buffer_.size();
    buffer_.resize(bufferSize + element.size() + 1);
    std::copy(element.begin(), element.end(), buffer_.begin() + bufferSize);
    buffer_.back() = detail::fastqHeaderValuesSeparator;
}

void FastqCreateTask::createHeader(SampleIndex::FastqOffsetsContainer::value_type offset,
                                   bool                                           filterFlag)
{
    const auto& bclBuffer = bclBuffers_[offset.first];

    const common::ControlFlag controlFlag = bclBuffer.controls_[offset.second].data_;

    // HEADER
    buffer_.push_back(detail::fastqHeaderStart);
    // instrument
    writeHeaderElement(flowcellInfo_.getInstrument());
    // run number
    writeHeaderElement(flowcellInfo_.getRunNumber());
    // flow cell ID
    writeHeaderElement(flowcellInfo_.getFlowcellId());

    // lane number
    BOOST_ASSERT(laneInfo_.getNumber() < 1000);

    if (laneInfo_.getNumber() >= 100)
    {
        buffer_.push_back(char((laneInfo_.getNumber() / 100) + '0'));
        int32_t remainder = laneInfo_.getNumber() % 100;
        buffer_.push_back(char((remainder / 10) + '0'));
        buffer_.push_back(char((remainder % 10) + '0'));
    }
    else if (laneInfo_.getNumber() >= 10)
    {
        buffer_.push_back(char((laneInfo_.getNumber() / 10) + '0'));
        buffer_.push_back(char((laneInfo_.getNumber() % 10) + '0'));
    }
    else
    {
        buffer_.push_back(char(laneInfo_.getNumber() + '0'));
    }

    buffer_.push_back(detail::fastqHeaderValuesSeparator);
    // tile number
    common::putUnsignedInteger(bclBuffer.tileInfo_->getNumber(), buffer_);
    buffer_.push_back(detail::fastqHeaderValuesSeparator);
    // x position
    common::putUnsignedInteger( bclBuffer.positions_.size() > offset.second
                              ? bclBuffer.positions_[offset.second].x_
                              : 0, buffer_ );
    buffer_.push_back(detail::fastqHeaderValuesSeparator);
    // y position
    common::putUnsignedInteger( bclBuffer.positions_.size() > offset.second
                              ? bclBuffer.positions_[offset.second].y_
                              : 0, buffer_ );
    // UMI
    if (!umiCycles_.empty())
    {
        buffer_.push_back(detail::fastqHeaderValuesSeparator);
        size_t i = 0;
        for (const auto& cycle : umiCycles_)
        {
            if (i > 0)
            {
                buffer_.push_back(detail::fastqHeaderUmiSeparator);
            }
            std::transform(
                FastqConstIterator(bclBuffer.bcls_.begin() + cycle.first, offset.second),
                FastqConstIterator(bclBuffer.bcls_.begin() + cycle.second, offset.second),
                std::back_inserter(buffer_),
                &convertBcl2FastqBase
            );
            ++i;
        }
    }
    buffer_.push_back(detail::fastqHeaderPositionsSeparator);

    // read number
    BOOST_ASSERT(readInfo_.getNumber() <= 9);
    buffer_.push_back(char(readInfo_.getNumber() + '0'));
    buffer_.push_back(detail::fastqHeaderValuesSeparator);
    // is-filtered
    buffer_.push_back(filterFlag ? 'N' : 'Y');
    buffer_.push_back(detail::fastqHeaderValuesSeparator);
    // control number
    common::putUnsignedInteger(controlFlag, buffer_);
    buffer_.push_back(detail::fastqHeaderValuesSeparator);
    // barcode
    if (bclBuffer.samples_[offset.second].second.componentsBegin() == bclBuffer.samples_[offset.second].second.componentsEnd())
    {
        common::putUnsignedInteger((laneInfo_.sampleInfosBegin()+sampleMetadata_.sampleIndex_)->getNumber(), buffer_);
    }
    else
    {
        bclBuffer.samples_[offset.second].second.output(buffer_);
    }

    buffer_.push_back(detail::fastqHeaderSeparator);
}

void FastqCreateTask::createBasesAndQualities(SampleIndex::FastqOffsetsContainer::value_type offset,
                                              io::GzipCompressor& compressor,
                                              size_t& trimmedCount)
{
    const auto& bclBuffer = bclBuffers_[offset.first];
    const FastqConstIterator begin(bclBuffer.bcls_.begin() + cycleIndex_, offset.second);

    // The last cycle we want to put in the FASTQ file
    const FastqConstIterator end(bclBuffer.bcls_.begin() + cycleIndexEnd_, offset.second);

    // BASES
    FastqConstIterator maskBegin = end;
    BOOST_FOREACH(const AdapterLocator& adapterLocator, std::make_pair(maskAdapters_.begin(), maskAdapters_.end()))
    {
        const FastqConstIterator adapterPos = adapterLocator.identifyAdapter(begin, end);
        maskBegin = std::min(maskBegin, adapterPos);
    }

    FastqConstIterator trimBegin = end;
    BOOST_FOREACH(const AdapterLocator& adapterLocator, std::make_pair(trimAdapters_.begin(), trimAdapters_.end()))
    {
        const FastqConstIterator adapterPos = adapterLocator.identifyAdapter(begin, end);
        trimBegin = std::min(trimBegin, adapterPos);
    }

    trimmedCount = end - std::min(trimBegin, maskBegin);
    if (readInfo_.isDataRead())
    {
        if ((trimBegin - begin) < FastqConstIterator::difference_type(minimumTrimmedReadLength_))
        {
            maskBegin = std::min(trimBegin, maskBegin);
            trimBegin = begin + minimumTrimmedReadLength_;
        }

        if ((maskBegin - begin) < FastqConstIterator::difference_type(maskShortAdapterReads_))
        {
            maskBegin = begin;
        }
    }

    const FastqConstIterator::difference_type maskedCount = trimBegin - maskBegin;
    FastqConstIterator basesEnd = std::min(maskBegin, trimBegin);
    size_t bufferSize = buffer_.size();

    size_t numBases = std::distance(begin, basesEnd);
    size_t numBasesPlusMask = numBases;
    if (maskedCount > 0)
    {
        numBasesPlusMask += maskedCount;
    }

    // Resize for bases + BasesSeparator + quality + QualitySeparator
    buffer_.resize(bufferSize + 2*numBasesPlusMask + 1 + detail::fastqBasesSeparator.size());
    std::vector<char>::iterator iter(buffer_.begin());
    std::advance(iter, bufferSize);
    if (generateReverseComplementFastqs_)
    {
        if (maskedCount > 0)
        {
            std::fill_n(iter, maskedCount, detail::fastqMaskedBase);
            std::advance(iter, maskedCount);
        }

        std::transform(
            FastqConstReverseIterator(basesEnd),
            FastqConstReverseIterator(begin),
            iter,
            &convertBcl2FastqBaseComplement
        );

        std::advance(iter, numBases);
    }
    else
    {
        std::transform(
            begin,
            basesEnd,
            iter,
            &convertBcl2FastqBase
        );

        std::advance(iter, numBases);
        if (maskedCount > 0)
        {
            std::fill_n(iter, maskedCount, detail::fastqMaskedBase);
            std::advance(iter, maskedCount);
        }
    }

    std::copy(detail::fastqBasesSeparator.begin(), detail::fastqBasesSeparator.end(), iter);
    std::advance(iter, detail::fastqBasesSeparator.size());

    // QUALITIES
    if (generateReverseComplementFastqs_)
    {
        if (maskedCount > 0)
        {
            std::fill_n(iter, maskedCount, detail::fastqMaskedQuality);
            std::advance(iter, maskedCount);
        }

        std::transform(FastqConstReverseIterator(basesEnd), FastqConstReverseIterator(begin), iter, &convertBcl2FastqQuality);
        std::advance(iter, numBases);
    }
    else
    {
        std::transform(begin, basesEnd, iter, &convertBcl2FastqQuality);

        std::advance(iter, numBases);
        if (maskedCount > 0)
        {
            std::fill_n(iter, maskedCount, detail::fastqMaskedQuality);
            std::advance(iter, maskedCount);
        }
    }
    *iter = detail::fastqQualitiesSeparator;

    std::streamsize bytesWritten = compressor.write(&*buffer_.begin(), buffer_.size());
    BCL2FASTQ_ASSERT_MSG( static_cast<FastqBuffer::FastqsContainer::value_type::value_type::value_type::size_type>(bytesWritten) == buffer_.size(),
                          "Only " << bytesWritten << " of " << buffer_.size() << " bytes have been written" );
}

FastqCreator::FastqCreator(
    common::ThreadVector::size_type threadsCount,
    StageMediator<InputBuffer> &inputMediator,
    StageMediator<OutputBuffer> &outputMediator,
    const layout::Layout &layout,
    const layout::LaneInfo &laneInfo,
    std::size_t maskShortAdapterReads,
    float adapterStringency,
    bool generateReverseComplementFastqs,
    bool includeNonPfClusters,
    bool createFastqsForIndexReads,
    bool useBgzf,
    int compressionLevel,
    bool findAdaptersWithSlidingWindow,
    FastqCreateTask::ConversionStats &summaryTileStats,
    stats::BarcodeHits &unknownBarcodeHits
)
: IntermediateStage<BclBufferVec, FastqBuffer>(threadsCount, "Fastq creating", inputMediator, outputMediator)
, layout_(layout)
, laneInfo_(laneInfo)
, threadsCount_(threadsCount)
, sampleIndex_(laneInfo.sampleInfosEnd()-laneInfo.sampleInfosBegin(), BclBuffer::clustersPerChunk_)
, maskShortAdapterReads_(maskShortAdapterReads)
, adapterStringency_(adapterStringency)
, generateReverseComplementFastqs_(generateReverseComplementFastqs)
, includeNonPfClusters_(includeNonPfClusters)
, createFastqsForIndexReads_(createFastqsForIndexReads)
, useBgzf_(useBgzf)
, compressionLevel_(compressionLevel)
, findAdaptersWithSlidingWindow_(findAdaptersWithSlidingWindow)
, summaryTileStats_(summaryTileStats)
, tileStats_()
, unknownBarcodesHits_(unknownBarcodeHits)
, unknownBarcodes_()
{
    size_t nonIndexReadLength = 0;
    size_t numNonIndexReads = 0;
    for (const auto& readInfo : laneInfo_.readInfos())
    {
        if (readInfo.isDataRead())
        {
            nonIndexReadLength = std::max(nonIndexReadLength, readInfo.cycleInfos().size());
            numNonIndexReads = std::max(numNonIndexReads, readInfo.getNumber());
        }
    }

    layout::LaneInfo::SampleInfosContainer::size_type sampleIndex = 0;
    BOOST_FOREACH (const layout::SampleInfo &sampleInfo, std::make_pair(laneInfo_.sampleInfosBegin(), laneInfo_.sampleInfosEnd()))
    {
        layout::BarcodesContainer::size_type barcodesCount = sampleInfo.getBarcodes().size();
        if( 0 == barcodesCount )
        {
            createTileStats( layout::BarcodeTranslationTable::SampleMetadata(sampleIndex),
                             nonIndexReadLength,
                             numNonIndexReads );
        } else {
            for( layout::BarcodesContainer::size_type barcodeIndex = 0;
                 barcodeIndex < barcodesCount;
                 ++barcodeIndex )
            {
                createTileStats( layout::BarcodeTranslationTable::SampleMetadata(sampleIndex,barcodeIndex),
                                 nonIndexReadLength,
                                 numNonIndexReads );
            }
        }
        ++sampleIndex;
    }

    for (common::ThreadVector::size_type threadNum = 0; threadNum < threadsCount_; ++threadNum)
    {
        tileStats_.push_back(summaryTileStats_);
        unknownBarcodes_.push_back(unknownBarcodesHits_);
    }

}


void FastqCreator::createTileStats(const layout::BarcodeTranslationTable::SampleMetadata &sampleMetadata,
                                   size_t nonIndexReadLength,
                                   size_t numNonIndexReads)
{
    layout::LaneInfo::TileInfosContainer::size_type tilesCount = laneInfo_.getTileInfos().size();
    for( layout::LaneInfo::TileInfosContainer::size_type tileIndex = 0;
         tileIndex < tilesCount;
         ++tileIndex )
    {
        summaryTileStats_.first.push_back(
                          FastqCreateTask::ConversionStats::first_type::value_type(
                                           stats::TileStats<stats::AllReadsStats>::Index(sampleMetadata, tileIndex)
                          )
        );
        summaryTileStats_.first.back()[stats::TileBarcodeStats::RAW].init(nonIndexReadLength, numNonIndexReads);
        summaryTileStats_.first.back()[stats::TileBarcodeStats::PF].init(nonIndexReadLength, numNonIndexReads);

        for (const auto& readInfo : laneInfo_.readInfos())
        {
            if (readInfo.isDataRead())
            {
                summaryTileStats_.second.push_back(
                                  FastqCreateTask::ConversionStats::second_type::value_type(
                                                   stats::TileStats<stats::ReadStats>::Index(sampleMetadata, tileIndex),
                                                   readInfo.getNumber()
                                  )
                );
            }
        }
    }
}

bool FastqCreator::preExecute()
{
    TaskQueue &taskQueue = this->getTaskQueue();
    FastqBuffer &outputBuffer = this->getOutputBuffer();
    const auto& inputBuffer = this->getInputBuffer();

    sampleIndex_.reset(inputBuffer);

    layout::LaneInfo::SampleInfosContainer::size_type sampleInfosCounts = std::distance(laneInfo_.sampleInfosBegin(), laneInfo_.sampleInfosEnd());

    layout::ReadInfosContainer::size_type readsCount = std::count_if(laneInfo_.readInfos().begin(),
                                                                     laneInfo_.readInfos().end(),
                                                                     [this] (const layout::ReadInfo& readInfo)
                                                                         { return !readInfo.cycleInfos().empty() &&
                                                                                  (createFastqsForIndexReads_ || readInfo.isDataRead()); });
    outputBuffer.fastqs_.resize(sampleInfosCounts);

    BCL2FASTQ_ASSERT_MSG(inputBuffer.front().bcls_.size() > 0, "No BCL cycles to process");
    BCL2FASTQ_LOG(common::LogLevel::DEBUG) << "Clusters/BCL: " << inputBuffer.front().bcls_.size() << std::endl;

    for( layout::LaneInfo::SampleInfosContainer::size_type sampleInfoIndex = 0;
         sampleInfoIndex < sampleInfosCounts;
         ++sampleInfoIndex )
    {
        size_t cycleIndex = 0;

        outputBuffer.fastqs_[sampleInfoIndex].resize(readsCount);
        layout::ReadInfosContainer::size_type readInfoIndex = 0;
        for (const auto& readInfo : laneInfo_.readInfos())
        {
            maskAdapters_.push_back(new boost::ptr_vector<AdapterLocator>());
            trimAdapters_.push_back(new boost::ptr_vector<AdapterLocator>());

            createAdapters(maskAdapters_.back(),
                           readInfo.maskAdaptersBegin(),
                           readInfo.maskAdaptersEnd());

            createAdapters(trimAdapters_.back(),
                           readInfo.trimAdaptersBegin(),
                           readInfo.trimAdaptersEnd());
            // For index reads, use unmasked cycles.
            size_t nextCycleIndex = cycleIndex + readInfo.getNumCyclesToLoad();

           // cycleIndex += readInfo.getBclBufferOffset();
            if (readInfo.shouldCreateFastqForRead(createFastqsForIndexReads_))
            {
                SampleIndex::FastqOffsetsContainer::size_type offsetsIndex = 0;
                const SampleIndex::FastqOffsetsContainer::const_iterator offsetsEnd = sampleIndex_.offsetsEnd(sampleInfoIndex);
                SampleIndex::FastqOffsetsContainer::const_iterator offset = sampleIndex_.offsetsBegin(sampleInfoIndex);

                outputBuffer.fastqs_[sampleInfoIndex][readInfoIndex].resize((offsetsEnd-offset + ClustersPerTask - 1) / ClustersPerTask);
                while(offset != offsetsEnd)
                {
                    BCL2FASTQ_ASSERT_MSG(offsetsEnd - offset > 0, "Invalid offset iterator");
                    const SampleIndex::FastqOffsetsContainer::const_iterator nextOffset =
                        offset
                        +
                        (static_cast<std::size_t>(offsetsEnd - offset) < ClustersPerTask ? offsetsEnd - offset : ClustersPerTask); 
                    taskQueue.addTask(new FastqCreateTask(
                        inputBuffer,
                        cycleIndex,
                        nextCycleIndex,//cycleIndex + readInfo.cycleInfos().size(),
                        layout_.getFlowcellInfo(),
                        laneInfo_,
                        readInfo,
                        offset,
                        nextOffset,
                        maskShortAdapterReads_,
                        adapterStringency_,
                        maskAdapters_.back(),
                        trimAdapters_.back(),
                        generateReverseComplementFastqs_,
                        includeNonPfClusters_,
                        useBgzf_,
                        compressionLevel_,
                        findAdaptersWithSlidingWindow_,
                        tileStats_,
                        unknownBarcodes_,
                        outputBuffer.fastqs_[sampleInfoIndex][readInfoIndex][offsetsIndex]
                    ));
                    ++offsetsIndex;
                    offset = nextOffset;
                }
                ++readInfoIndex;
            }
            
            cycleIndex = nextCycleIndex;
        }
    }

    return true;
}


bool FastqCreator::postExecute()
{
    for (common::ThreadVector::size_type threadNum = 0; threadNum < threadsCount_; ++threadNum)
    {
        detail::transferStats< FastqCreateTask::ConversionStats::first_type >( tileStats_[threadNum].first,
                                                                               summaryTileStats_.first );
        detail::transferStats< FastqCreateTask::ConversionStats::second_type >( tileStats_[threadNum].second,
                                                                                summaryTileStats_.second );
        unknownBarcodesHits_.merge( unknownBarcodes_[threadNum] );
        unknownBarcodes_[threadNum].reset();
    }

    return true;
}

void FastqCreator::createAdapters(boost::ptr_vector<AdapterLocator>& adapters,
                                  layout::ReadInfo::AdaptersContainer::const_iterator begin,
                                  layout::ReadInfo::AdaptersContainer::const_iterator end) const
{
    size_t minimumTrimmedReadLength = laneInfo_.getMinimumTrimmedReadLength();
    BOOST_FOREACH (const layout::ReadInfo::AdaptersContainer::value_type& adapter, std::make_pair(begin, end))
    {
        findAdaptersWithSlidingWindow_ ?
            adapters.push_back(new AdapterLocatorSlidingWindow(adapter, adapterStringency_)) :
            adapters.push_back(new AdapterLocatorWithIndels(adapter, adapterStringency_, minimumTrimmedReadLength));
    }
}

} // namespace conversion
} // namespace bcl2fastq


