/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file FastqWriter.cpp
 *
 * \brief Implementation of FASTQ writer.
 *
 * \author Marek Balint
 */


#include <utility>

#include <boost/thread/locks.hpp>
#include <boost/format.hpp>
#include <boost/filesystem.hpp>
#include <boost/foreach.hpp>

#include "common/Debug.hh"
#include "conversion/FastqWriter.hh"


namespace bcl2fastq {


namespace conversion {


namespace detail {


} // namespace detail


FastqWriteTask::FastqFileStateInfo::FastqFileStateInfo()
: allocationFlag_(false)
, fastqFileStatePtr_()
{
}

FastqWriteTask::FastqFileStateInfo::FastqFileStateInfo(data::FastqFile::FastqFileStatePtr fastqFileStatePtr)
: allocationFlag_(false)
, fastqFileStatePtr_(fastqFileStatePtr)
{
}


FastqWriteTask::FastqWriteTask(
    FastqFilesContainer &fastqFiles,
    FastqFileStateInfo &fastqFileStateInfo,
    const FastqBuffer::FastqsContainer::value_type::value_type &inputBuffer
)
: Task()
, fastqFiles_(fastqFiles)
, fastqFileStateInfo_(fastqFileStateInfo)
, inputBuffer_(inputBuffer)
{
}

bool FastqWriteTask::execute(common::ThreadVector::size_type threadNum)
{
    BOOST_FOREACH (const FastqBuffer::FastqsContainer::value_type::value_type::value_type &read,
                   std::make_pair(inputBuffer_.begin(), inputBuffer_.end()))
    {
        data::FastqFile &fastqFile = fastqFiles_.at(threadNum);
        fastqFile.write(*fastqFileStateInfo_.fastqFileStatePtr_, &*read.begin(), read.size());
    }

    return true;
}

FastqWriter::FastqWriter(
    common::ThreadVector::size_type threadsCount,
    StageMediator<InputBuffer> &inputMediator,
    const layout::Layout &layout,
    const layout::LaneInfo &laneInfo,
    bool createFastqsForIndexReads,
    const boost::filesystem::path& outputDir,
    bool noLaneSplitting
)
: SinkStage<FastqBuffer>(threadsCount, "Fastq writing", inputMediator)
, layout_(layout)
, emptyFileDeleter_()
, fastqFiles_()
, fastqFileStates_()
{
    data::FastqFile fastqFile;

    BOOST_FOREACH (const layout::SampleInfo &sampleInfo, std::make_pair(laneInfo.sampleInfosBegin(), laneInfo.sampleInfosEnd()))
    {
        fastqFileStates_.push_back(FastqFileStatesContainer::value_type());
        for (const auto& readInfo : laneInfo.readInfos())
        {
            if (!readInfo.shouldCreateFastqForRead(createFastqsForIndexReads))
            {
                continue;
            }
            std::ostringstream fileName;
            fileName << (noLaneSplitting ?
                boost::format("%s_S%d_%c%d_001.fastq.gz")
                    % sampleInfo.getName()
                    % sampleInfo.getNumber()
                    % (readInfo.isIndexRead() ? 'I' : 'R')
                    % readInfo.getNumber() :
                boost::format("%s_S%d_%s_%c%d_001.fastq.gz")
                    % sampleInfo.getName()
                    % sampleInfo.getNumber()
                    % laneInfo.getDirName().string()
                    % (readInfo.isIndexRead() ? 'I' : 'R')
                    % readInfo.getNumber())
            ;

            boost::filesystem::path fastqPath(outputDir);
            if (sampleInfo.hasProject())
            {
                fastqPath /= sampleInfo.getProject();
                if( sampleInfo.getId() != sampleInfo.getName() )
                {
                    fastqPath /= sampleInfo.getId();
                }
            }

            boost::filesystem::path filePath(fastqPath / boost::filesystem::path(fileName.str()));
            emptyFileDeleter_.addFile(filePath);

            boost::filesystem::path dirPath = filePath.parent_path();
            if (!dirPath.empty())
            {
                boost::filesystem::create_directories(dirPath);
            }

            fastqFileStates_.back().push_back(
                FastqWriteTask::FastqFileStateInfo(
                    fastqFile.openFile(filePath,
                                       std::ios_base::out | std::ios_base::binary | (noLaneSplitting && laneInfo.getNumber() > 1 ?
                                           std::ios_base::app :
                                           std::ios_base::trunc ))
                )
            );
        }
    }

    for (common::ThreadVector::size_type threadNumber = 0; threadNumber < threadsCount; ++threadNumber)
    {
        fastqFiles_.push_back(new data::FastqFile());
    }
}


bool FastqWriter::preExecute()
{
    TaskQueue &taskQueue = this->getTaskQueue();
    FastqBuffer &inputBuffer = this->getInputBuffer();

    FastqBuffer::FastqsContainer::value_type::size_type sampleIndex = 0;
    BOOST_FOREACH (const FastqBuffer::FastqsContainer::value_type &sample, std::make_pair(inputBuffer.fastqs_.begin(), inputBuffer.fastqs_.end()))
    {
        FastqBuffer::FastqsContainer::value_type::value_type::size_type readIndex = 0;
        BOOST_FOREACH (const FastqBuffer::FastqsContainer::value_type::value_type &read, std::make_pair(sample.begin(), sample.end()))
        {
            taskQueue.addTask(new FastqWriteTask(
                fastqFiles_,
                fastqFileStates_.at(sampleIndex).at(readIndex),
                read
            ));

            ++readIndex;
        }
        ++sampleIndex;
    }

    return true;
}

bool FastqWriter::postExecute()
{
    return true;
}

FastqWriter::EmptyFileDeleter::~EmptyFileDeleter()
{
    for (const boost::filesystem::path& filePath : filePaths_)
    {
        if(boost::filesystem::exists(filePath) && boost::filesystem::file_size(filePath) == 0)
        {
            boost::filesystem::remove(filePath);
        }
    }
}


} // namespace conversion


} // namespace bcl2fastq


