/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file Stage.cpp
 *
 * \brief Implementation of a processing stage.
 *
 * \author Marek Balint
 */


//#define WITH_CPU_TIMER


#include <boost/thread/locks.hpp>
#include <boost/format.hpp>
#if defined(WITH_CPU_TIMER)
#   include <boost/timer/timer.hpp>
#endif

#include "common/Timer.hh"
#include "common/Logger.hh"
#include "conversion/Stage.hh"


namespace bcl2fastq {


namespace conversion {


Stage::Stage(common::ThreadVector::size_type threadsCount,
             const std::string&              stageName)
: preExecuteTime_(0)
, postExecuteTime_(0)
, mutex_()
, threadVector_(threadsCount)
, taskQueue_()
, stageName_(stageName)
{
}

Stage::~Stage()
{
    BCL2FASTQ_LOG(common::LogLevel::DEBUG) << "Total time spent in pre-execution of " << stageName_ << ": " << preExecuteTime_/1000000 << " s" << std::endl;
    BCL2FASTQ_LOG(common::LogLevel::DEBUG) << "Total time spent in post-execution of " << stageName_ << ": " << postExecuteTime_/1000000 << " s" << std::endl;
}

void Stage::run()
{
    try
    {
        BCL2FASTQ_LOG(common::LogLevel::DEBUG) << "Starting " << stageName_ << " stage" << std::endl;

        std::size_t executionCounter = 0;
#       if defined(WITH_CPU_TIMER)
            const long double sec = 1000000000.0L;
            long double wallTime = 0;
            long double userTime = 0;
            long double kernTime = 0;
#       endif
        while (this->getWork())
        {
#           if defined(WITH_CPU_TIMER)
               boost::timer::cpu_timer cpuTimer;
               cpuTimer.start();
#           endif

            BCL2FASTQ_LOG(common::LogLevel::DEBUG)
                << stageName_ << " stage resuming work: "
                << executionCounter << std::endl;

            {
                common::Timer timer(preExecuteTime_);
                if (!this->preExecute())
                {
                    break;
                }
            }

            BCL2FASTQ_LOG(common::LogLevel::DEBUG)
                << stageName_ << " stage executing with "
                << taskQueue_.getOutstandingTasksCount()
                << " outstanding tasks" << std::endl;

            threadVector_.execute(boost::bind(
                &Stage::executeThreadFunction,
                this,
                _1
            ));
            BCL2FASTQ_LOG(common::LogLevel::DEBUG) << stageName_ << " execution done" << std::endl;

            // The tasks are complete. Delete them.
            taskQueue_.clear();

            {
                common::Timer timer(postExecuteTime_);
                if (!this->postExecute())
                {
                    break;
                }
            }

#           if defined(WITH_CPU_TIMER)
                cpuTimer.stop();
                boost::timer::cpu_times t = cpuTimer.elapsed();
                wallTime = ((wallTime * executionCounter) + (t.wall / sec)) / (executionCounter + 1);
                userTime = ((userTime * executionCounter) + (t.user / sec)) / (executionCounter + 1);
                kernTime = ((kernTime * executionCounter) + (t.system / sec)) / (executionCounter + 1);

                BCL2FASTQ_LOG(common::LogLevel::DEBUG)
                    << stageName_ << " stage submitting work: " << executionCounter << ":"
                    << " wall=" << (boost::format("%07.4f") % (t.wall / sec)) << "s"
                    << " user=" << (boost::format("%07.4f") % (t.user / sec)) << "s"
                    << " kern=" << (boost::format("%07.4f") % (t.system / sec)) << "s"
                    << " avgwall=" << (boost::format("%07.4f") % wallTime) << "s"
                    << " avguser=" << (boost::format("%07.4f") % userTime) << "s"
                    << " avgkern=" << (boost::format("%07.4f") % kernTime) << "s"
                    << std::endl
                ;
#           else
                BCL2FASTQ_LOG(common::LogLevel::DEBUG)
                    << stageName_ << " stage submitting work: "
                    << executionCounter << std::endl;
#           endif

            this->submitWork();

            BCL2FASTQ_LOG(common::LogLevel::DEBUG) << stageName_ << " stage work finished" << std::endl;

            ++executionCounter;
        }
        this->finishWork();
    }
    catch (...)
    {
        BCL2FASTQ_LOG(common::LogLevel::DEBUG) << stageName_ << " stage terminating" << std::endl;
        this->terminate();
        throw;
    }

    BCL2FASTQ_LOG(common::LogLevel::DEBUG) << stageName_ << " stage done" << std::endl;
}

TaskQueue & Stage::getTaskQueue()
{
    return taskQueue_;
}

void Stage::executeThreadFunction(common::ThreadVector::size_type threadNum)
{
    while (Task *task = this->getNextTask())
    {
        if (!task->execute(threadNum))
        {
            break;
        }
    }
}

Task * Stage::getNextTask()
{
    boost::unique_lock<boost::mutex> lock(mutex_);
    Task *task = taskQueue_.getTask();
    if (task && !task->isSticky())
    {
        taskQueue_.advanceTask();
    }
    return task;
}

bool Stage::getWork()
{
    // default implementation: just return true
    return true;
}

void Stage::submitWork()
{
    // default implementation: do nothing
}

void Stage::finishWork()
{
    // default implementation: do nothing
}


} // namespace task


} // namespace bcl2fastq


