/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file BclFile.cpp
 *
 * \brief Implementation of BCL file.
 *
 * \author Marek Balint
 * \author Mauricio Varea
 * \author Aaron Day
 */


#include <boost/filesystem/operations.hpp>
#include <boost/format.hpp>
#include "common/Debug.hh"
#include "common/Logger.hh"
#include "common/SystemCompatibility.hh"
#include "data/BclFile.hh"

namespace bcl2fastq {


namespace data {


BclFile::BclFile()
: FileReaderBase("", false)
, BinaryFileReader()
, currentFilePosition_()
, gzipDecompressor_(io::GzipDecompressor::PageSize)
, isCompressed_(true)
{
}

void BclFile::getAndVerifyFileName(const boost::filesystem::path& inputDir,
                                   common::LaneNumber             laneNumber,
                                   common::CycleNumber            cycleNumber,
                                   bool                           ignoreErrors,
                                   boost::filesystem::path&       fileName)
{
    fileName = boost::filesystem::path(
        inputDir
        /
        boost::filesystem::path((boost::format("L%03d") % laneNumber).str())
        /
        boost::filesystem::path((boost::format("%04d.bcl.bgzf") % cycleNumber).str())
    );

    if (!boost::filesystem::exists(fileName))
    {
        std::string errStr("Unable to find BCL file: '" + fileName.string() + "'");
        if (ignoreErrors)
        {
            BCL2FASTQ_LOG(common::LogLevel::WARNING) << errStr << std::endl;
        }
        else
        {
            BOOST_THROW_EXCEPTION(common::IoError(ENOENT, errStr));
        }

        fileName = boost::filesystem::path();
    }
}

void BclFile::getAndVerifyFileName(const boost::filesystem::path& inputDir,
                                   common::LaneNumber             laneNumber,
                                   common::TileNumber             tileNumber,
                                   common::CycleNumber            cycleNumber,
                                   bool                           ignoreErrors,
                                   boost::filesystem::path&       fileName)
{
    std::string sample = (boost::format("s_%d_%d") % laneNumber % tileNumber).str();

    fileName = boost::filesystem::path(
        inputDir
        /
        boost::filesystem::path((boost::format("L%03d") % laneNumber).str())
        /
        boost::filesystem::path((boost::format("C%d.1") % cycleNumber).str())
        /
        boost::filesystem::path((boost::format("%s.bcl.gz") % sample).str())
    );

    if (!boost::filesystem::exists(fileName))
    {
        fileName.replace_extension();

        if (!boost::filesystem::exists(fileName))
        {
            std::string errStr("Unable to find BCL file for '" + sample + "' in: " + fileName.remove_filename().string());
            if (ignoreErrors)
            {
                BCL2FASTQ_LOG(common::LogLevel::WARNING) << errStr << std::endl;
            }
            else
            {
                BOOST_THROW_EXCEPTION(common::IoError(ENOENT, errStr));
            }

            fileName = boost::filesystem::path();
        }
    }
}

void BclFile::openFile(
    const boost::filesystem::path& inputDir,
    common::LaneNumber             laneNumber,
    common::CycleNumber            cycleNumber,
    bool                           ignoreErrors
)
{
    boost::filesystem::path fileName;
    getAndVerifyFileName(inputDir,
                         laneNumber,
                         cycleNumber,
                         ignoreErrors,
                         fileName);

    openFile(fileName,
             ignoreErrors,
             true);
}

void BclFile::openFile(
    const boost::filesystem::path& inputDir,
    common::LaneNumber             laneNumber,
    common::TileNumber             tileNumber,
    common::CycleNumber            cycleNumber,
    bool                           ignoreErrors
)
{
    boost::filesystem::path fileName;
    getAndVerifyFileName(inputDir,
                         laneNumber,
                         tileNumber,
                         cycleNumber,
                         ignoreErrors,
                         fileName);

    openFile(fileName,
             ignoreErrors,
             (fileName.extension() == ".gz"));
}

void BclFile::openFile(
    const boost::filesystem::path &path,
    bool ignoreErrors,
    bool compressed,
    common::ClustersCount defaultClustersCount
)
{
    static const std::string errMsgBegin = "Ignoring file open failure for bcl file '";

    try
    {
        isCompressed_ = compressed;
        init(path,
             ignoreErrors,
             defaultClustersCount);

        if(!openFileBuf())
        {
            return;
        }

        gzipDecompressor_.flush();
        gzipDecompressor_.reset();

        if (!readHeader(clustersCount_))
        {
            return;
        }

        /// \todo Refactoring: Do not determine endianness in runtime, let CMake to do it.
        if (!common::isLittleEndian())
        {
            clustersCount_ = common::swapEndian(clustersCount_);
        }
        currentFilePosition_ = fileBuf_.pubseekoff(0, std::ios_base::cur);

        BCL2FASTQ_LOG(common::LogLevel::INFO)
            << "Opened BCL file '" << path_.string() << "' with "
            << clustersCount_ << " clusters" << std::endl;
    }
    CATCH_AND_IGNORE(std::ios_base::failure, errMsgBegin)
    CATCH_AND_IGNORE(io::ZlibError, errMsgBegin)
    CATCH_AND_IGNORE_ALL(errMsgBegin)
}

std::streamsize BclFile::read(std::istream &is, char *buffer, std::streamsize size)
{
    if( isCompressed_ )
    {
        return gzipDecompressor_.read(is, buffer, size);
    }
    else
    {
        is.read(buffer, size);
        return is.gcount();
    }

    return 0;
}

std::streamsize BclFile::read(char *targetBuffer, std::streamsize targetSize)
{
    static const std::string errMsgBegin = "Ignoring read failure for bcl file '";

    if (!validateCondition(this->isOpen(), "File is not open")) { return 0; }
    if (!validateCondition(fileBuf_.is_open(), "File is not open")) { return 0; }

    std::streamsize bytesRead = -1;
    try
    {
        std::istream is(&fileBuf_);

        bytesRead = read(is, targetBuffer, targetSize);
        while ((bytesRead < targetSize))
        {
            std::streamsize bytesRead2 = read(is, targetBuffer+bytesRead, targetSize-bytesRead);
            if (bytesRead2 != -1)
            {
                bytesRead += bytesRead2;
            } else {
                break;
            }
        }
    }
    CATCH_AND_IGNORE(io::ZlibError, errMsgBegin)
    CATCH_AND_IGNORE(common::IoError, errMsgBegin)
    CATCH_AND_IGNORE(std::ios_base::failure, errMsgBegin)
    CATCH_AND_IGNORE_ALL(errMsgBegin)
    
    return bytesRead;
}

void BclFile::seek(std::streamsize compressedOffset, std::streamsize uncompressedOffset)
{
    static const std::string errMsgBegin = "Ignoring seek failure for: '";

    BOOST_ASSERT(isCompressed_);

    try
    {
        std::istream is(&fileBuf_);
        is.seekg(compressedOffset);
        is.sync();
        std::vector<char> targetBuffer(uncompressedOffset);

        gzipDecompressor_.reset();
        gzipDecompressor_.flush();
        std::streamsize bytesRead = gzipDecompressor_.read(is, &*(targetBuffer.begin()), uncompressedOffset);

        int errnum = errno;
        if (bytesRead != uncompressedOffset)
        {
            BOOST_THROW_EXCEPTION(common::InputDataError(errnum, (boost::format("Unable to seek to given position of BCL file '%s': compressed_offset=%d uncompressed_offset=%d bytes_read=%d") % path_.string() % compressedOffset % uncompressedOffset % bytesRead).str()));
        }
    }
    CATCH_AND_IGNORE(common::IoError, errMsgBegin)
    CATCH_AND_IGNORE_ALL(errMsgBegin)
}


} // namespace data


} // namespace bcl2fastq


