/**
* BCL to FASTQ file converter
* Copyright (c) 2007-2015 Illumina, Inc.
*
* This software is covered by the accompanying EULA
* and certain third party copyright/licenses, and any user of this
* source file is bound by the terms therein.
*
* \file ControlFile.cpp
*
* \brief Implementation of control file.
*
* \author Aaron Day
*/

#include "data/ControlFile.hh"

#include <boost/filesystem/operations.hpp>

namespace bcl2fastq
{
namespace data
{

ControlFile::ControlFile(const boost::filesystem::path& filePath,
                         bool                           ignoreErrors /*= false*/)
    : FileReaderBase(filePath,
                     ignoreErrors)
    , BinaryAllClustersFileReader(FileReaderBase::getPath(),
                                  ignoreErrors)
{
    readFile();
}

bool ControlFile::readHeader()
{
    if (!BinaryAllClustersFileReader::readHeader(clustersCount_))
    {
        return false;
    }

    if (clustersCount_ == 0)
    {
        Header header;
        if (!BinaryAllClustersFileReader::readHeader(header))
        {
            return false;
        }

        if (!validateHeader(header))
        {
            return false;
        }

        clustersCount_ = header.clustersCount_;
    }

    return true;
}

bool ControlFile::validateHeader(const ControlFile::Header& header)
{
    int errnum = errno;
    if (header.version_ != Header::VERSION)
    {
        const uint32_t version = header.version_;
        BOOST_THROW_EXCEPTION(common::InputDataError(errnum, (boost::format("Corrupted header of control file '%s': header_version=%d") % path_.string() % version).str()));
    }

    return true;
}

void ControlFile::readRecords()
{
    readClusters(buffer_,
                 clustersCount_);

    bufferPosition_ = buffer_.begin();
}

std::size_t ControlFile::read(
    ControlFile::Record *targetBuffer,
    std::size_t targetSize
)
{
    if (!validateCondition(this->isOpen(), "File is not open")) { return 0; }

    if (!validateCondition((buffer_.end() - bufferPosition_) >= 0, "Invalid buffer size.")) { return 0; }
    if (!validateCondition((buffer_.end() - bufferPosition_) % sizeof(Record) == 0, "Invalid buffer size.")) { return 0; }
    const std::size_t recordsToRead = std::min(
        static_cast<std::size_t>(buffer_.end() - bufferPosition_) / sizeof(Record),
        targetSize
    );

    const Record * const begin = reinterpret_cast<const Record * const>(&*bufferPosition_);
    const Record * const end = begin + recordsToRead;

    std::copy(begin, end, targetBuffer);
    bufferPosition_ += (recordsToRead * sizeof(Record));

    return recordsToRead;
}

const uint32_t ControlFile::Header::VERSION = 2;


boost::shared_ptr<ControlFile> ControlFileFactory::createControlFile(
    const boost::filesystem::path& intensitiesDir,
    common::LaneNumber             laneNumber,
    common::TileNumber             tileNumber,
    bool                           ignoreErrors)
{
    boost::filesystem::path fileName((boost::format("s_%d_%d.control") % laneNumber % tileNumber).str());
    boost::filesystem::path fileNameZeroPadded((boost::format("s_%d_%04d.control") % laneNumber % tileNumber).str());
    boost::filesystem::path laneDir((boost::format("L%03d") % laneNumber).str());

    boost::filesystem::path filePath = intensitiesDir / laneDir / fileName;

    if (boost::filesystem::exists(filePath))
    {
        return boost::shared_ptr<ControlFile>(new ControlFile(filePath,
                                                              ignoreErrors));
    }

    filePath = intensitiesDir / laneDir / fileNameZeroPadded;

    if (boost::filesystem::exists(filePath))
    {
        return boost::shared_ptr<ControlFile>(new ControlFile(filePath,
                                                              ignoreErrors));
    }

    return boost::shared_ptr<ControlFile>();
}

} // namespace data
} // namespace bcl2fastq

