/**
* BCL to FASTQ file converter
* Copyright (c) 2007-2015 Illumina, Inc.
*
* This software is covered by the accompanying EULA
* and certain third party copyright/licenses, and any user of this
* source file is bound by the terms therein.
*
* \file FileReader.cpp
*
* \brief Implementation of FileReader.
*
* \author Aaron Day
*/

#include "data/FileReader.hh"

#include "common/Debug.hh"
#include "common/Exceptions.hh"


namespace bcl2fastq
{
namespace data
{

FileReaderBase::FileReaderBase(const boost::filesystem::path& path,
                               bool                           ignoreErrors,
                               common::ClustersCount          defaultClustersCount /*= 0*/)
    : path_(path)
    , ignoreErrors_(ignoreErrors)
    , clustersCount_(defaultClustersCount)
{
}

FileReaderBase::~FileReaderBase()
{
}

const boost::filesystem::path& FileReaderBase::getPath() const
{
    validateCondition(this->isOpen(), "File is not open");
    return path_;
}

common::ClustersCount FileReaderBase::getClustersCount() const
{
    validateCondition(this->isOpen(), "File is not open");
    return clustersCount_;
}

bool FileReaderBase::validateCondition(bool condition, const std::string& warningMsg) const
{
    if (!condition)
    {
        if (ignoreErrors_)
        {
            BCL2FASTQ_LOG(common::LogLevel::WARNING) << warningMsg << std::endl;
        }
        else
        {
            BCL2FASTQ_ASSERT_MSG(condition, warningMsg);
        }
    }

    return condition;
}

FileReader::FileReader(std::ios_base::openmode        openFlags,
                       const boost::filesystem::path& path,
                       bool                           ignoreErrors,
                       common::ClustersCount          defaultClustersCount /*=0*/)
    : FileReaderBase(path,
                     ignoreErrors,
                     defaultClustersCount)
    , fileBuf_(openFlags)
{
}

FileReader::FileReader(std::ios_base::openmode openFlags)
    : FileReaderBase("", false)
    , fileBuf_(openFlags)
{
}

FileReader::~FileReader()
{
}

void FileReader::init(const boost::filesystem::path& path,
                      bool                           ignoreErrors,
                      common::ClustersCount          defaultClustersCount /*=0*/)
{
    path_ = path;
    ignoreErrors_ = ignoreErrors;
    clustersCount_ = defaultClustersCount;
}

bool FileReader::openFileBuf()
{
    errno = 0;

    fileBuf_.reopen(getPath(), io::FileBufWithReopen::FadviseFlags::SEQUENTIAL_ONCE);
    int errnum = errno;
    if (!fileBuf_.is_open())
    {
        fileBuf_.reset();
        boost::format errFormat(boost::format("Unable to open %s file '%s' for reading")
                                     % getFileTypeStr()
                                     % getPath().string());
        if (ignoreErrors_)
        {
            BCL2FASTQ_LOG(common::LogLevel::WARNING)
                << errFormat
                << ":" << std::strerror(errnum) << " (" << errnum << ")" << std::endl;

            return false;
        }
        else
        {
            BOOST_THROW_EXCEPTION(common::IoError(errnum, errFormat.str()));
        }
    }

    return true;
}

BinaryFileReader::BinaryFileReader()
    : FileReaderBase("", false)
    , FileReader(std::ios_base::in | std::ios_base::binary)
{
}

BinaryFileReader::BinaryFileReader(const boost::filesystem::path& path,
                                   bool                           ignoreErrors,
                                   common::ClustersCount          defaultClustersCount /*=0*/)
    : FileReaderBase(path,
                     defaultClustersCount)
    , FileReader(std::ios_base::in | std::ios_base::binary,
                 path,
                 ignoreErrors,
                 defaultClustersCount)
{
}

BinaryFileReader::~BinaryFileReader()
{
}

std::streamsize BinaryFileReader::read(char *targetBuffer, std::streamsize targetSize)
{
    return io::read(fileBuf_,
                    targetBuffer,
                    targetSize);
}

void BinaryFileReader::logError(std::streamsize bytesRead,
                                std::streamsize bytesExpected) const
{
    int errnum = errno;

    boost::format errFormat(boost::format(
        getFileTypeStr() + " file '%s' truncated: bytes_read=%d bytes_expected=%d") %
        this->getPath().string() % bytesRead % bytesExpected);

    if (ignoreErrors_)
    {
        BCL2FASTQ_LOG(common::LogLevel::WARNING)
            << errFormat
            << ":" << std::strerror(errnum) << " (" << errnum << ")" << std::endl;
    }
    else
    {
        BOOST_THROW_EXCEPTION(common::InputDataError(errnum,
                                                     errFormat.str()));
    }
}

BinaryAllClustersFileReader::BinaryAllClustersFileReader(const boost::filesystem::path& path,
                                                         bool                           ignoreErrors)
    : FileReaderBase(path,
                     ignoreErrors)
    , BinaryFileReader(path,
                       ignoreErrors)
{
}

bool BinaryAllClustersFileReader::readClusters(std::vector<char>& buffer,
                                               uint32_t           clustersCount)
{
    std::size_t bufferSize = getRecordBytes() * clustersCount;
    buffer.resize(bufferSize);
    std::streamsize bytesRead =
        io::read(fileBuf_,
                 reinterpret_cast<char *>(&*buffer.begin()),
                 bufferSize);

    if (bytesRead != static_cast<std::streamsize>(bufferSize))
    {
        logError(bytesRead,
                 bufferSize);

        buffer.resize(bytesRead > 0 ? bytesRead - bytesRead % getRecordBytes() : 0);

        return false;
    }

    return true;
}

void BinaryAllClustersFileReader::validateHeader()
{
    // Let derived classes decide if they want to implement this
}

void BinaryAllClustersFileReader::readFile()
{
    try
    {
        if (!openFileBuf())
        {
            return;
        }

        if (!readHeader())
        {
            return;
        }

        readRecords();
    }
    CATCH_AND_IGNORE(common::IoError, "Ignoring read failure for " + getFileTypeStr() + " file '")
    CATCH_AND_IGNORE(std::ios_base::failure, "Ignoring read failure for " + getFileTypeStr() + " file '")
    CATCH_AND_IGNORE_ALL("Ignoring read failure for " + getFileTypeStr() + " file '")
}

} // namespace data
} // namespace bcl2fastq

