/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file FilterFile.cpp
 *
 * \brief Implementation of filter file.
 *
 * \author Marek Balint
 * \author Mauricio Varea
 */


#include <boost/format.hpp>
#include <boost/filesystem/operations.hpp>

#include "io/FileBufWithReopen.hh"
#include "data/FilterFile.hh"


namespace bcl2fastq {


namespace data {


FilterFileBase::FilterFileBase(const boost::filesystem::path& filePath,
                               bool                           ignoreErrors)
    : FileReaderBase(filePath,
                     ignoreErrors)
    , BinaryAllClustersFileReader(filePath,
                                  ignoreErrors)
{
}

bool FilterFileBase::readHeader()
{
    if (!BinaryAllClustersFileReader::readHeader(clustersCount_))
    {
        return false;
    }

    if (clustersCount_ == 0)
    {
        Header header;
        if (!BinaryAllClustersFileReader::readHeader(header))
        {
            return false;
        }

        if (!validateHeader(header))
        {
            return false;
        }

        clustersCount_ = header.clustersCount_;
    }

    return true;
}

bool FilterFileBase::validateHeader(const FilterFile::Header& header)
{
    int errnum = errno;
    if (header.version_ != getHeaderVersion())
    {
        const uint32_t version = header.version_;
        BOOST_THROW_EXCEPTION(common::InputDataError(errnum, (boost::format("Corrupted header of filter file '%s': header_version=%d") % path_.string() % version).str()));

        return false;
    }

    return true;
}

void FilterFileBase::readRecords()
{
    readClusters(buffer_,
                 clustersCount_);

    bufferPosition_ = buffer_.begin();
}

FilterFile::FilterFile(const boost::filesystem::path& filePath,
                       bool                           ignoreErrors)
    : FileReaderBase(filePath,
                     ignoreErrors)
    , FilterFileBase(filePath,
                     ignoreErrors)
{
    readFile();
}

std::size_t FilterFile::read(
    FilterFile::Record *targetBuffer,
    std::size_t targetSize
)
{
    if (!validateCondition(this->isOpen(), "File is not open")) { return 0; }
    if (!validateCondition((buffer_.end() - bufferPosition_) >= 0, "Invalid buffer size.")) { return 0; }
    if (!validateCondition((buffer_.end() - bufferPosition_) % sizeof(Record) == 0, "Invalid buffer size.")) { return 0; }

    const std::size_t recordsToRead = std::min(
        static_cast<std::size_t>(buffer_.end() - bufferPosition_) / sizeof(Record),
        targetSize
    );

    const Record * const begin = reinterpret_cast<const Record * const>(&*bufferPosition_);
    const Record * const end = begin + recordsToRead;

    std::copy(begin, end, targetBuffer);
    bufferPosition_ += (recordsToRead * sizeof(Record));

    return recordsToRead;
}


bool FilterFile::doesFileExist(const boost::filesystem::path& inputDir,
                               bool                           aggregateTilesFlag,
                               common::LaneNumber             laneNumber,
                               common::TileNumber             tileNumber,
                               boost::filesystem::path&       filePath)
{
    if( aggregateTilesFlag ) {
        filePath =
            inputDir
            / boost::filesystem::path((boost::format("L%03d") % laneNumber).str())
            / boost::filesystem::path((boost::format("s_%d.filter") % laneNumber).str());
    } else {
        boost::filesystem::path fileName((boost::format("s_%d_%d.filter") % laneNumber % tileNumber).str());
        boost::filesystem::path fileNameZeroPadded((boost::format("s_%d_%04d.filter") % laneNumber % tileNumber).str());
        boost::filesystem::path laneDir((boost::format("L%03d") % laneNumber).str());

        filePath = inputDir / laneDir / fileName;

        if (!boost::filesystem::exists(filePath))
        {
            filePath = inputDir / fileName;
        }

        if (!boost::filesystem::exists(filePath))
        {
            filePath = inputDir / laneDir / fileNameZeroPadded;
        }

        if (!boost::filesystem::exists(filePath))
        {
            filePath = inputDir / fileNameZeroPadded;
        }
    }

    return boost::filesystem::exists(filePath);
}


} // namespace data


} // namespace bcl2fastq


