/**
 * BCL to FASTQ file converter
 * Copyright (c) 2007-2015 Illumina, Inc.
 *
 * This software is covered by the accompanying EULA
 * and certain third party copyright/licenses, and any user of this
 * source file is bound by the terms therein.
 *
 * \file FileExistenceVerifier.cpp
 *
 * \brief Implementation of FileExistenceVerifier.
 *
 * \author Aaron Day
 */

#include "layout/FileExistenceVerifier.hh"

#include "layout/LaneInfo.hh"
#include "data/FilterFile.hh"
#include "data/PositionsFile.hh"
#include "data/BclFile.hh"

namespace bcl2fastq
{
namespace layout
{

void FileExistenceVerifier::verifyAllFilesExist(const boost::filesystem::path& inputDir,
                                                const boost::filesystem::path& intensitiesDir,
                                                const LaneInfo&                laneInfo,
                                                bool                           aggregateTilesFlag,
                                                bool                           isPatternedFlowcell,
                                                bool                           ignoreMissingBcls,
                                                bool                           ignoreMissingFilters,
                                                bool                           ignoreMissingPositions)
{
    if (aggregateTilesFlag)
    {
        verifyFilesExist(inputDir,
                         intensitiesDir,
                         laneInfo,
                         aggregateTilesFlag,
                         isPatternedFlowcell,
                         ignoreMissingBcls,
                         ignoreMissingFilters,
                         ignoreMissingPositions);
    }
    else
    {
        for (const auto& tileInfo : laneInfo.getTileInfos())
        {
            verifyFilesExist(inputDir,
                             intensitiesDir,
                             laneInfo,
                             aggregateTilesFlag,
                             isPatternedFlowcell,
                             ignoreMissingBcls,
                             ignoreMissingFilters,
                             ignoreMissingPositions,
                             tileInfo.getNumber());
        }
    }
}

void FileExistenceVerifier::verifyFilesExist(const boost::filesystem::path& inputDir, 
                                             const boost::filesystem::path& intensitiesDir,
                                             const LaneInfo&                laneInfo,
                                             bool                           aggregateTilesFlag,
                                             bool                           isPatternedFlowcell,
                                             bool                           ignoreMissingBcls,
                                             bool                           ignoreMissingFilters,
                                             bool                           ignoreMissingPositions,
                                             const common::TileNumber       tileNumber /*= 0*/)
{
    for (const auto& readInfo : laneInfo.readInfos())
    {
        if (!ignoreMissingBcls)
        {
            for (const layout::CycleInfo &cycleInfo : readInfo.cycleInfos())
            {
                verifyBclFileExists(inputDir,
                                    aggregateTilesFlag,
                                    laneInfo.getNumber(),
                                    cycleInfo.getNumber(),
                                    tileNumber);
            }
        }

        if (!ignoreMissingFilters)
        {
            verifyFilterFileExists(inputDir,
                                   aggregateTilesFlag,
                                   laneInfo.getNumber(),
                                   tileNumber,
                                   readInfo.getNumber());
        }
    }

    if (!ignoreMissingPositions)
    {
        verifyPositionsFileExists(intensitiesDir,
                                  aggregateTilesFlag,
                                  isPatternedFlowcell,
                                  laneInfo.getNumber(),
                                  tileNumber);
    }
}

void FileExistenceVerifier::verifyBclFileExists(const boost::filesystem::path& inputDir,
                                                bool                           aggregateTilesFlag,
                                                common::LaneNumber             laneNumber,
                                                common::CycleNumber            cycleNumber,
                                                common::TileNumber             tileNumber)
{
    boost::filesystem::path fileName;
    if (aggregateTilesFlag)
    {
        data::BclFile::getAndVerifyFileName(inputDir,
                                            laneNumber,
                                            cycleNumber,
                                            false,
                                            fileName);
    }
    else
    {
        data::BclFile::getAndVerifyFileName(inputDir,
                                            laneNumber,
                                            tileNumber,
                                            cycleNumber,
                                            false,
                                            fileName);
    }
}

void FileExistenceVerifier::verifyFilterFileExists(const boost::filesystem::path& inputDir,
                                                   bool                           aggregateTilesFlag,
                                                   common::LaneNumber             laneNumber,
                                                   common::TileNumber             tileNumber,
                                                   common::ReadNumber             readNumber)
{
    boost::filesystem::path filePath;
    if (!(data::FilterFile::doesFileExist(inputDir,
                                          aggregateTilesFlag,
                                          laneNumber,
                                          tileNumber,
                                          filePath)))
    {
        throwException("filter",
                       aggregateTilesFlag,
                       laneNumber,
                       tileNumber);
    }
}

void FileExistenceVerifier::verifyPositionsFileExists(const boost::filesystem::path& intensitiesDir,
                                                      bool                           aggregateTilesFlag,
                                                      bool                           isPatternedFlowcell,
                                                      common::LaneNumber             laneNumber,
                                                      common::TileNumber             tileNumber)
{
    if (!data::PositionsFileFactory::doesFileExist(intensitiesDir,
                                                   aggregateTilesFlag,
                                                   isPatternedFlowcell,
                                                   laneNumber,
                                                   tileNumber))
    {
        throwException("positions",
                       aggregateTilesFlag,
                       laneNumber,
                       tileNumber);
    }
}

void FileExistenceVerifier::throwException(const std::string& fileType,
                                           bool               aggregateTilesFlag,
                                           common::LaneNumber laneNumber,
                                           common::TileNumber tileNumber)
{
    std::stringstream str;
    str << "Unable to find " << fileType << " file for lane: " << laneNumber;
    if (!aggregateTilesFlag)
    {
        str << " and tile: " << tileNumber;
    }
    str << std::endl;

    BOOST_THROW_EXCEPTION(common::IoError(ENOENT, str.str()));
}

} // namespace layout
} // namespace bcl2fastq

